class twitch_cassandra (
  $cluster_name,
  $cluster_seeds,
  $auto_bootstrap         = false,
  $compaction_throughput  = $twitch_cassandra::params::compaction_throughput,
  $heap_newsize           = $twitch_cassandra::params::heap_newsize,
  $hinted_handoff_enabled = $twitch_cassandra::params::hinted_handoff_enabled,
  $java_distribution      = $twitch_cassandra::params::java_distribution,
  $java_version           = $twitch_cassandra::params::java_version,
  $listen_address         = $twitch_cassandra::params::listen_address,
  $max_heap_size          = $twitch_cassandra::params::max_heap_size,
  $opscenter_host         = $twitch_cassandra::params::opscenter_host,
  $package_name           = $twitch_cassandra::params::package_name,
  $package_version        = $twitch_cassandra::params::package_version,
  $backup                 = $twitch_cassandra::params::backup,
  $restore                = $twitch_cassandra::params::restore,
  $include_repair         = $twitch_cassandra::params::include_repair,
  $include_opscenter      = $twitch_cassandra::params::include_opscenter,
  $include_monitoring     = $twitch_cassandra::params::include_monitoring,
  $backupday              = $twitch_cassandra::params::backupday,
  $backuphour             = $twitch_cassandra::params::backuphour,
  $data_file_directories  = $twitch_cassandra::params::data_file_directories,
  $commitlog_directory    = $twitch_cassandra::params::commitlog_directory,
  $manage_java            = true,
  $disable_auth           = $twitch_cassandra::params::disable_auth,
) inherits twitch_cassandra::params {

  validate_string($include_repair,$include_opscenter,$include_monitoring)

  if $include_monitoring == 'true' {
    include twitch_cassandra::monitoring
  }

  if $include_opscenter == 'true' {
    include twitch_cassandra::opscenter
  }

  if $include_repair == 'true' {
    include twitch_cassandra::repair
  }
  #
  # package { 'unzip':
  #   ensure => 'present',
  # }
  #
  package { 'jq':
    ensure => 'present',
  }

  realize(Package['unzip'])

  if $manage_java {
    $cassandra_require = [Class['java']]
    class { 'java':
      distribution => $java_distribution,
      version      => $java_version,
    }
  }
  else {
    $cassandra_require = []
  }

  Apt::Source['datastax-mirror'] ->
  Exec['apt-get update cassandra'] ->
  Class['cassandra']

  # Setup the repo to pull the package from
  apt::source { 'datastax-mirror':
      location    => 'http://aptly.internal.justin.tv/internal/datastax',
      release     => 'stable',
      repos       => 'main',
      include_src => false,
      notify      => Exec['apt-get update cassandra']
  }

  exec { "apt-get update cassandra":
    command => 'apt-get update',
    refreshonly => true,
  }

  if ($::ec2_instance_id != "") {
    $endpoint_snitch = "Ec2Snitch"
    $install_topology = false
  }
  else {
    $endpoint_snitch = "PropertyFileSnitch"
    $install_topology = true
  }

  class { 'cassandra':
    cluster_name           => $cluster_name,
    compaction_throughput  => $compaction_throughput,
    heap_newsize           => $heap_newsize,
    hinted_handoff_enabled => $hinted_handoff_enabled,
    listen_address         => $listen_address,
    max_heap_size          => $max_heap_size,
    package_name           => $package_name,
    require                => $cassandra_require,
    rpc_address            => '0.0.0.0',
    seeds                  => $cluster_seeds,
    version                => $package_version,
    include_repo           => false,
    endpoint_snitch        => $endpoint_snitch,
    data_file_directories  => $data_file_directories,
    commitlog_directory    => $commitlog_directory,
    disable_auth           => $disable_auth
  }

  # TODO change this. dont want business logic in puppet files (templates + hiera would be better)
  if $install_topology {
    file { '/etc/cassandra/cassandra-topology.properties':
      # contains mappings of hosts to racks and datacenters
      ensure  => present,
      owner   => 'cassandra',
      group   => 'cassandra',
      mode    => '0644',
      source  => "puppet:///modules/${module_name}/cassandra-topology/${cluster_name}.properties",
      require => File['/etc/cassandra'],  # defined in modules/cassandra
      notify  => Class['cassandra::service'],
    }
  }



  $local_version = $twitch_cassandra::package_version

  case $local_version {
    /2.1.2-1/: { $jammlinksource = 'jamm-0.2.8.jar' }
    default:   { $jammlinksource = 'jamm-0.3.0.jar' }
  }

  if $twitch_cassandra::backup {
    user { 'cassbackup':
      gid => 'cassandra'
    }
    file { '/usr/local/bin/cass-backup.py':
      ensure  => present,
      owner   => 'cassbackup',
      group   => 'cassandra',
      mode    => '0700',
      source  => "puppet:///modules/${module_name}/cass-backup.py",
      require => [Package[dsc21], User['cassbackup']]
    }
    twitch_cassandra::cronbackup {"cassbackupjob": backuphour => $backuphour, backupday => $backupday}
  }

  if $twitch_cassandra::restore {
    include twitch_cassandra::restorestaging
  }

  file { "$cass_lib_dir/jamm.jar":
    ensure   => 'link',
    target   => "$cass_lib_dir/$jammlinksource",
    require  => Class['cassandra']
  }

}
