# Creates an SSL certificate from Sandstorm.
# Parameters:
#   title              - Hostname that the cert is for.
#   sandstorm_priv_key - Sandstorm key of the private key for the cert
#  Use one of:
#   sandstorm_cert_key - Sandstorm key of the text of the certificate itself
#   cert_text          - Full text of the cert, if you're willing to put it in
#                        puppet directly.
#   cert_file          - puppet:/// path to cert file.
#
#  Optional:
#   cert_folder        - Optional. Directory to store the cert in.
#   priv_folder        - Optional. Directory to store the private key in.
#   notifies           - An array of strings, services to notify when the cert
#                        or key changes. Used only when defining certs in hiera.
#
# Usage:
#
# twitch_certificate { 'some.website.com':
#   sandstorm_cert_key => 'WILDCARD_INTERNAL_JUSTIN_TV_CERT',
#   sandstorm_priv_key => 'team/ssl_certs/production/some.website.com',
# }
#
# or:
# twitch_certificate { 'some.website.com':
#   cert_text => template("${module_name}/somecert.erb")
#   sandstorm_priv_key => 'team/ssl_certs/production/some.website.com',
# }
#
# or:
# twitch_certificate { 'some.website.com':
#   cert_file => "puppet:///modules/meseeks/lookatme.pem"
#   sandstorm_priv_key => 'team/ssl_certs/production/some.website.com',
# }
#
# You probably also want to notify => Service['apache2'] or Service['nginx'] too
#
# Files will end up, by default, at:
# Cert - /etc/ssl/certs/$hostname.pem
# Key  - /etc/ssl/private/$hostname.key
#
# Twitch_core also can create these resources directly from hiera if you wish.
# Use the "ssl_certificates" key, so:
#
# ssl_certificates:
#   sandstorm_cert_key: 'systems/ssl_certs/staging/wildcard.dev.us-west2.justin.tv'
#   sandstorm_priv_key: 'systems/ssl_keys/staging/wildcard.dev.us-west2.justin.tv'
#   notifies:
#     - nginx
#
# 'notifies' is a solution to the problem of not being able to define
# dependencies in hiera. Anything in this array will be added to
# notify => Service[$notifies] on the ssl cert resource.


define twitch_certificate (
  $hostname    = $title,

  $sandstorm_cert_key = undef,
  $cert_text          = undef,
  $cert_file          = undef,

  $sandstorm_priv_key = undef,

  $cert_folder        = '/etc/ssl/certs',
  $priv_folder        = '/etc/ssl/private',

  $notifies           = [],
) {

  validate_string($name)
  validate_string($sandstorm_priv_key)

  # Build an array out of the parameters, so we can cleanly validate that
  # exactly one of them were provided.
  $validate_params = delete_undef_values([
    $sandstorm_cert_key,
    $cert_text,
    $cert_file,
  ])

  if size($validate_params) == 0 {
    fail('twitch_certificate expects one of sandstorm_cert_key, cert_text, or cert_file, but none were provided.')
  }
  if size($validate_params) > 1 {
    fail('twitch_certificate expects one of sandstorm_cert_key, cert_text, or cert_file, but multiple were provided.')
  }

  $cert_filename = "${cert_folder}/${hostname}.pem"
  $key_filename  = "${priv_folder}/${hostname}.key"

  $real_notifies = $notifies ? {
    undef => undef,
    default => Service[$notifies],
  }

  if $sandstorm_cert_key {
    validate_string($sandstorm_cert_key)

    twitch_sandstorm_agent::template { "${hostname}-cert":
      destination => $cert_filename,
      key         => $sandstorm_cert_key,
      notify      => $real_notifies,
    }
  } elsif $cert_text {
    validate_string($cert_text)

    file { $cert_filename:
      ensure  => 'present',
      mode    => '0644',
      owner   => 'root',
      group   => 'root',
      content => $cert_text,
      notify  => $real_notifies,
    }
  } else {
    validate_string($cert_file)

    file { $cert_filename:
      ensure => 'present',
      mode   => '0644',
      owner  => 'root',
      group  => 'root',
      source => $cert_file,
      notify => $real_notifies,
    }
  }

  twitch_sandstorm_agent::template { "${hostname}-key":
    destination => $key_filename,
    key         => $sandstorm_priv_key,
    notify      => $real_notifies,
  }
}
