#!/usr/bin/env ruby
require 'augeas'
require 'optparse'
require 'ostruct'

# Use augueas to set resolver affinity to the resolver in the current
# AZ. If this fails, not the end of the world since this is run as an
# exit hook to dhclient.

# Barrel-shift the resolver order for secondary and tertiary resolvers.
$az_primary = {
  "us-west-2a" => "10.192.67.85",
  "us-west-2b" => "10.192.69.171",
  "us-west-2c" => "10.192.75.166",
}
$az_secondary = {
  "us-west-2a" => "10.192.69.171",
  "us-west-2b" => "10.192.75.166",
  "us-west-2c" => "10.192.67.85",
}
$az_tertiary = {
  "us-west-2a" => "10.192.75.166",
  "us-west-2b" => "10.192.67.85",
  "us-west-2c" => "10.192.69.171",
}

$resolvers = $az_primary.values.sort

options = OpenStruct.new
options.local_resolver = false
OptionParser.new do |opts|
  opts.on("-l", "--local-resolver", "Use local resolver") do |o|
    options.local_resolver = o
  end
  opts.parse!(ARGV)
end

az = %x{ec2metadata --availability-zone}
az.strip!

primary = $az_primary[az]
secondary =  $az_secondary[az]
tertiary = $az_tertiary[az]

# Likelihood is that all fail, but this is belt and suspenders
# territory.
if primary == nil or secondary == nil or tertiary == nil
  STDERR.puts("Error looking up resolver for #{az}")
  exit(0)
end

nameservers = Array.new
Augeas::open do |aug|
  [1,2,3].each do |i|
    ns = aug.get("/files/etc/resolv.conf/nameserver[#{i}]")
    nameservers << ns unless ns == nil
  end
  nameservers.sort!
  if nameservers != $resolvers
    # We expected the list of resolvers in resolv.conf to match the
    # list hardcoded. Since the lists didn't match, we might be in a
    # different VPC with different resolver options, or the resolver
    # configuration has changed. In that case we'll just exit.
    STDERR.puts("Error: resolv.conf resolvers doesn't match: expected #{$resolvers.inspect}, got #{nameservers.inspect}")
    exit(0)
  end
  if options.local_resolver
    aug.set("/files/etc/resolv.conf/nameserver[1]", "127.0.0.1")
    aug.set("/files/etc/resolv.conf/nameserver[2]", primary)
    aug.set("/files/etc/resolv.conf/nameserver[3]", secondary)
    aug.set("/files/etc/resolv.conf/nameserver[4]", tertiary)
  else
    aug.set("/files/etc/resolv.conf/nameserver[1]", primary)
    aug.set("/files/etc/resolv.conf/nameserver[2]", secondary)
    aug.set("/files/etc/resolv.conf/nameserver[3]", tertiary)
  end
  unless aug.save
    STDERR.puts("Failed to save changes to resolv.conf")
    exit(0)
  end
end

