#!/usr/bin/env ruby
require 'yaml'

run_status_schema = Struct.new(:puppet_version, :config_version, :changes_total, :failed_events, :failed_resources, :failed_restarts, :last_run, :failed_run_count) do
    def initialize(puppet_version=nil, config_version=nil, changes_total=nil, failed_events=nil, failed_resources=nil, failed_restarts=nil, last_run=nil, failed_run_count=0)
        super
    end
end

def check_history()
    if File.exist?('/tmp/puppet_error_log.yaml')
        history = YAML.load_file('/tmp/puppet_error_log.yaml')
        if history.respond_to? :value then history.value else history end
    else
        nil
    end
end

def write_history(history)
    File.open('/tmp/puppet_error_log.yaml','w:BINARY') { |f| f.write YAML.dump(history)}
end

# set time since last run and consecutive run failure bounds
stale_min = 60
max_run_fail = 2

critical = []
warning = []

unless File.exists?('/var/lib/puppet/state/last_run_summary.yaml')
    puts "UNKONWN: puppet last run summary not found!"
    exit 3
end

puppet_status = YAML.load_file('/var/lib/puppet/state/last_run_summary.yaml')
history = check_history()
now = Time.now
last_run = Time.at(puppet_status['time']['last_run'])

unless puppet_status.key? 'events' and puppet_status.key? 'changes'
    puts "WARNING: Puppet v#{version_puppet} last run was a noop"
    exit 1
end

if puppet_status['events']['failure'] == 0 and puppet_status['resources']['failed'] == 0 and puppet_status['resources']['failed_to_restart'] and not last_run < now - (60 * stale_min)
    if history
        File.delete('/tmp/puppet_error_log.yaml')
    end
    puts "OK: Puppet v#{puppet_status['version']['puppet']} executed catalog version #{puppet_status['version']['config']} and made #{puppet_status['changes']['total']} changes."
    exit 0
end

if last_run < now - (60 * stale_min)
    diff = now - last_run
    critical << "Puppet run stale by #{diff}s. Last run: #{last_run}"
end

unless history
    history = run_status_schema.new()
end

if puppet_status['events']['failure'] > 0 or puppet_status['resources']['failed'] == 0 or puppet_status['resources']['failed_to_restart']
    if history['failed_run_count'] >= max_run_fail-1
        critical << "Consective run failues exceeds set threshold!"
    end
    warning << "Last run had #{puppet_status['events']['failure']} failed events, #{puppet_status['resources']['failed']} failed resources, and #{puppet_status['resources']['failed_to_restart']} failed service restarts."
end

unless history['last_run'] == last_run.to_i
    history['puppet_version'] = puppet_status['version']['puppet']
    history['config_version'] = puppet_status['version']['config']
    history['changes_total'] = puppet_status['changes']['total']
    history['failed_events'] = puppet_status['events']['failure']
    history['failed_resources'] = puppet_status['resources']['failed']
    history['failed_restarts'] = puppet_status['resources']['failed_to_restart']
    history['last_run'] = puppet_status['time']['last_run']
    history['failed_run_count'] += 1
    write_history(history)
end

if critical.count > 0
    errors = (critical + warning).join('; ')
    puts "CRITICAL: #{errors}"
    exit 2
elsif warning.count > 0
    errors = warning.join('; ')
    puts "WARNING: #{errors}"
    exit 1
else
    puts "UNKNOWN!"
    exit 3
end
