#
# consul_service_fqdn_lookup.rb
#

$LOAD_PATH.unshift File.dirname(__FILE__)

require 'libconsul'

module Puppet::Parser::Functions
  newfunction(:consul_service_fqdn_lookup, :type => :rvalue, :doc => <<-EOS
Lookup fqdns for all service hosts for a datacenter

*Examples:*

    consul_service_fqdn_lookup('service', {'dc' => 'sjc02'})

All optional arguments must be passed in as an hash.

Optional Arguments:

    consul_api_endpoint: Api server and port to query.
    dc: Datacenter to lookup key in. By default 'sfo01'.
    EOS
  ) do |args|
    raise(Puppet::ParseError, 'consul_service_fqdn_lookup(): Wrong number of arguments ' +
          "given (#{args.size} for 1)") if args.size < 1

    service = args[0]
    opt = args[1] || {}

    # Defaulting to consul.internal because this function will be used to
    # populate 'start_join' list for consul agents.
    # Can't query localhost:8500 if consul agent not running because it needs to know
    # what cluster to join. Chicken + egg scenario.
    consul_api_endpoint = opt['consul_api_endpoint'] || 'consul.internal.justin.tv'
    dc = opt['dc'] || 'sfo01'

    query = ["dc=#{dc}"]
    service_catalog_url = URI::join("http://#{consul_api_endpoint}/v1/catalog/service/", service)
    service_catalog_url.query = query.join('&')

    service_catalog_response = consul_api_call(service_catalog_url)
    service_nodes = service_catalog_response.map { |node_meta| "#{node_meta['Node']}" }

    service_node_fqdns = []
    service_nodes.each do |node_name|
      node_services_url = URI::join("http://#{consul_api_endpoint}/v1/catalog/node/", node_name)
      # reuse same query since same datacenter
      node_services_url.query = query.join('&')

      node_services_response = consul_api_call(node_services_url)
      node_services = node_services_response['Services']
      if node_services.key?('nodeinfo')
        fqdn_tag = node_services['nodeinfo']['Tags'].select { |tag| tag.start_with?('fqdn=') }[0]
        service_node_fqdns.push(fqdn_tag.split('=')[1])
      elsif /^([a-z\d\.-]+)\.(justin|twitch)\.tv$/.match(node_name)
        # node_name is fqdn
        service_node_fqdns.push(node_name)
      else
        # node_name is host name
        service_node_fqdns.push("#{node_name}.#{dc}.justin.tv")
      end
    end

    # sort fqdns to prevent consul from unnecessary service refreshes if returned order changes
    return service_node_fqdns.sort
  end
end
