# twitch_core is the module which gets included on all Puppet nodes
# It manages directories, repositories, packages, and authentication
class twitch_core {
  $package_provider = $::osfamily ? {
    'RedHat' => 'yum',
    'Debian' => 'apt',
  }

  anchor { 'twitch_core::begin': } ->

  # Configure the network properly
  class { 'twitch_core::network::dhcp': } ->

  # directories have no dependencies
  class { 'twitch_core::directories::home_users': } ->
  class { 'twitch_core::directories::opt': } ->
  class { 'twitch_core::directories::opt_twitch': } ->
  class { 'twitch_core::directories::opt_virtualenvs': } ->
  class { 'twitch_core::directories::var_chroot': } ->

  # repositories also have no dependencies
  # these should be rewritten to use the open source modules more closely
  class { "twitch_core::repositories::${package_provider}": } ->
  class { 'twitch_core::repositories::python': } ->
  class { 'twitch_core::repositories::ruby': } ->

  # packages are configured after repositories
  class { 'twitch_core::packages::authentication': } ->
  class { 'twitch_core::packages::common': } ->
  class { 'twitch_core::packages::syslog': } ->
  class { 'twitch_core::packages::consul': } ->
  class { 'twitch_core::packages::gcc': } ->
  class { 'twitch_core::packages::kernel': } ->
  class { 'twitch_core::packages::locales': } ->
  class { 'twitch_core::packages::mail': } ->
  class { 'twitch_core::packages::nrpe': } ->
  class { 'twitch_core::packages::nagios': } ->
  class { 'twitch_core::packages::ntp': } ->
  class { 'twitch_core::packages::puppet': } ->
  class { 'twitch_core::packages::python': } ->
  class { 'twitch_core::packages::sysstat': } ->
  class { 'twitch_core::packages::virtual': } ->
  class { 'twitch_core::packages::monit': } ->

  # authentication is configured after packages
  class { 'twitch_core::authentication::ldap': } ->
  class { 'twitch_core::authentication::pam': } ->
  class { 'twitch_core::authentication::sudoers': } ->
  class { 'twitch_core::authentication::unscd': } ->
  class { 'twitch_core::authentication::ssh': } ->
  class { 'twitch_core::authentication::sshd': } ->

  # metric collection services
  class { 'twitch_core::metrics::diamond': } ->

  # logging framework
  class { 'twitch_core::logging::setup': } ->
  class { 'twitch_core::logging::default': } ->   # Basic rules
  class { 'twitch_core::logging::local3': } ->    # local3 => proc -> /var/log/jtv/proc.log
  class { 'twitch_core::logging::local4': } ->    # local4 => same as local3 + centralized remote syslog
  class { 'twitch_core::logging::remote': } ->    # sends things to remote syslog infra
  class { 'twitch_core::logging::puppet': } ->    # handle puppet-agent logs
  class { 'twitch_core::logging::consul': } ->    # handle consul logs
  class { 'twitch_core::logging::filebeat': } ->  # set up filebeat for log shipping

  anchor { 'twitch_core::end': }

  # Log the puppetmaster the client is connecting to
  notify { 'identify_puppetmaster_message':
    message => "This client, ${::twitch_uuid}::${::fqdn}, is connecting to ${::servername}",
  }

  # Install inventory agent
  include twitch_mgdb::agent

  # If we're running in a non-master environment, log a big ugly warning about
  # puppet being turned off. In an ideal world we have nagios checks on all of
  # these servers, so running from a different environment for too long will
  # set off warning bells.
  if $::environment != 'production' {
    stage {'dev-warning': }
    Stage['main'] -> Stage['dev-warning']

    class {'twitch_core::warning':
      stage => 'dev-warning',
    }
  }
}
