# class twitch_core::logging::filebeat setups default central logging
class twitch_core::logging::filebeat (
  $logstash_hosts = ['syslog-ingest.prod.us-west2.justin.tv'],
  $logstash_port  = 5044,
  $output_tag     = 'syslog_elasticsearch',
) {
  # We want to have Filebeat attempt to use TLS when we know puppet agent
  # has run, and has a trusted TLS certificate with the remote CA
  # we can verify that the puppet agent has a trusted key by looking in
  # the "special" fact delivered from facter:
  # https://docs.puppet.com/puppet/3.8/reference/lang_facts_and_builtin_vars.html#special-variables-added-by-puppet
  # we specifically handle the case where this is not a hash,
  # and therefore not addressable as one
  if is_hash($::trusted) {
    if $::trusted['authenticated'] == 'remote' {
      $use_tls = true
    } else {
      $use_tls = false
    }
  } else {
    $use_tls = false
  }

  $_base_log_tags = delete_undef_values([
    "role_${::twitch_role}",
    "environment_${::twitch_environment}",
    $::twitch_team ? {
      undef => undef,
      default => "team_${::twitch_team}",
    },
  ])

  # Replace dashes with underscores in tags so they get indexed as a whole
  # instead of by component.
  $_log_tags = regsubst($_base_log_tags, '-', '_', 'G')

  $_cert_dir = $::osfamily ? {
    'Debian' => '/etc/ssl/certs',
    'RedHat' => '/etc/pki/tls/certs',
    'amazon' => '/etc/pki/tls/certs',
  }

  if $use_tls {
    $_tls_opts = {
      # We include both CA certs here, because servers using puppet
      # (vs puppet-ci) need to be able to verify as well.
      'certificate_authorities' => [
        "${_cert_dir}/puppet-ci-ca.pem",
        "${_cert_dir}/puppet-ca.pem",
      ],
      'certificate'             => "/var/lib/puppet/ssl/certs/${::fqdn}.pem",
      'certificate_key'         => "/var/lib/puppet/ssl/private_keys/${::fqdn}.pem",
    }
  } else {
    $_tls_opts = undef
  }

  class { '::filebeat':
    registry_file => '/var/lib/filebeat',
    logging       => {
      level     => 'warning',
      to_syslog => false,
      to_files  => true,
      files     => {
        path      => '/var/log/filebeat',
        name      => 'filebeat.log',
        keepfiles => 7,
      }
    },
    shipper       => {
      # Name defaults to the hostname, which isn't fully qualified.
      'name' => $::fqdn,
      'tags' => $_log_tags,
    },
    outputs       => {
      'logstash' => delete_undef_values({
        'hosts'       => sort($logstash_hosts),
        'port'        => $logstash_port,
        'loadbalance' => true,
        'tls'         => $_tls_opts,
      }),
    }
  }

  ::filebeat::prospector { 'syslogs':
    paths    => [
      '/var/log/syslog',
      '/var/log/auth.log',
      '/var/log/daemon.log',
      '/var/log/mail.log',
    ],
    doc_type => 'syslog',
    fields   => {
      'output' => $output_tag,
    }
  }

  # Restarts Filebeat daily.  Put into place to workaround bug where Filebeat
  # holds log files open after they've been deleted.
  cron { "filebeat":
    user    => "root",
    hour    => fqdn_rand(24, 'filebeat_hour_seed'),
    minute  => fqdn_rand(60, 'filebeat_minute_seed'),
    command => "service filebeat restart > /dev/null 2>&1",
  }
}
