# class twitch_core::packages::consul - installs and configures consul
class twitch_core::packages::consul (
  $version      = $twitch_core::packages::consul_params::version,
  $package_version      = $twitch_core::packages::consul_params::package_version,
  $purge_config = $twitch_core::packages::consul_params::purge_config,
) inherits twitch_core::packages::consul_params {

  $statsd_host = hiera('twitch_statsd_host')
  $statsd_port = hiera('twitch_statsd_port')

  validate_string($version)
  validate_bool($purge_config)
  validate_array($twitch_core::packages::consul_params::start_join)
  validate_string($twitch_core::packages::consul_params::datacenter)
  validate_hash($twitch_core::packages::consul_params::ports)
  validate_string($twitch_core::packages::consul_params::arch)
  validate_string($twitch_core::packages::consul_params::os)
  validate_string($statsd_host)
  validate_re("$statsd_port", '^[0-9]+$')

  if $::osfamily == 'Debian' {
    # internal package name for Debian
    $_install_version = "${version}.${package_version}+${::lsbdistcodename}"
  } else {
    # for non-Debian we use vanilla version numbers
    $_install_version = "${version}-1"
  }

  class { '::consul':
    install_method   => 'package',
    package_ensure   => $_install_version,
    purge_config_dir => $purge_config,
    bin_dir          => '/usr/bin',
    config_hash      => {
      'advertise_addr'       => $::ipaddress,
      'bind_addr'            => '0.0.0.0',
      'data_dir'             => '/var/lib/consul',
      'datacenter'           => $twitch_core::packages::consul_params::datacenter,
      'disable_remote_exec'  => true,
      'disable_update_check' => true,
      'enable_syslog'        => true,
      'log_level'            => 'INFO',
      'node_name'            => $twitch_core::packages::consul_params::node_name,
      'ports'                => $twitch_core::packages::consul_params::ports,
      'server'               => false,
      'start_join'           => $twitch_core::packages::consul_params::start_join,
      'statsd_addr'          => "${statsd_host}:${statsd_port}",
      'syslog_facility'      => 'daemon',
    }
  }

  # The 'nodeinfo' service is a fake service which is used to report a node's
  # FQDN to Consul. This is here for legacy reasons, because we do report nodes
  # by FQDN to Consul (see 'node_name' above). Without this fake service, we
  # would be unable to use the Twitch deploy system, as this is how it selects
  # its targets.
  consul::service {'nodeinfo':
    tags => [
      "fqdn=${::fqdn}",
      "lsbdistcodename=${::lsbdistcodename}",
    ],
  }
}
