# Configure Nagios clients
class twitch_core::packages::nagios (
  $load_thresholds   = '-w 2.55,2.55 -c 2.80,2.80',
  $memory_thresholds = '-w 80:50 -c 90:70',
  $disk_thresholds   = '-w 25% -c 15%',
  $extra_hostgroups  = [],
  $host_template     = 'linux-server',
  $service_template  = 'generic-service',
  $host_contact      = undef
) {
  # Register the host in PuppetDB
  $all_hostgroups = unique(concat([$::twitch_role], $extra_hostgroups))
  # $::nagios_nic should be a fact manually put into /etc/facter/facts.d/
  if $::twitch_nagios_nic != undef {
    $nagios_ip = inline_template("<%= scope.lookupvar('::ipaddress_${::twitch_nagios_nic}') -%>")
  }
  else {
    $nagios_ip = $::ipaddress
  }
  @@nagios_host { $::fqdn:
    ensure             => present,
    address            => $nagios_ip,
    contacts           => $host_contact,
    max_check_attempts => '5',
    use                => $host_template,
    hostgroups         => join($all_hostgroups, ','),
    tag                => "team_${::twitch_team}_${::twitch_environment}",
  }

  # Turns ['a', 'b'] into ['a#fqdn', 'b#fqdn'], see twitch_nagios::hostgroup
  # docs for details.
  $hostgroup_labels = suffix($all_hostgroups, "#${::fqdn}")
  @@twitch_nagios::hostgroup { $hostgroup_labels:
    export => true,
    tag    => "team_${::twitch_team}_${::twitch_environment}",
  }

  # Define some basic checks that'll run on every host
  file {'/opt/nagios_checks':
    ensure  => 'directory',
    owner   => 'root',
    group   => 'nagios',
    recurse => true,
    mode    => '0755',
    source  => 'puppet:///modules/twitch_core/nrpe_checks',
    # We need the nagios group before this'll work
    require => Package[$::nrpe::package_name],
  }

  twitch_nagios::service {'cpu':
    is_nrpe     => true,
    command     => "check_load -r ${load_thresholds}",
    description => 'CPU Load',
    use         => $service_template,
  }

  twitch_nagios::service {'disk':
    is_nrpe     => true,
    command     => "check_disk ${disk_thresholds} -r /dev/ -C -w 50% -c 35% -p /boot -i \"/srv/node/.*\"",
    description => 'Disk space',
    use         => $service_template,
  }

  twitch_nagios::service {'mem':
    is_nrpe     => true,
    command     => "/opt/nagios_checks/check_memswap.pl ${memory_thresholds}",
    description => 'Memory',
    use         => $service_template,
  }

  twitch_nagios::service {'puppet':
    is_nrpe     => true,
    # This script needs sudo to read /var/lib/puppet/state/last_run_summary.yaml
    nrpe_sudo   => true,
    command     => '/opt/nagios_checks/check_puppet.rb',
    description => 'Puppet',
    use         => $service_template,
  }
  if $::ipaddress_bond0 {
    twitch_nagios::service {'bond':
      is_nrpe     => true,
      command     => '/opt/nagios_checks/check_bond',
      description => 'Bond',
      use         => $service_template,
    }
  }

  if str2bool($::hardware_raid_enabled) or str2bool($::software_raid_enabled) {
    twitch_nagios::service {'raid':
      is_nrpe     => true,
      command     => '/opt/nagios_checks/check_raid.pl',
      description => 'Raid',
      use         => $service_template,
    }
  } else {
    twitch_nagios::service {'raid':
      ensure      => 'absent',
      command     => '/bin/true',
      description => 'remove me',
      is_nrpe     => true,
      use         => $service_template,
    }
  }
}
