# Configures the Puppet agent's server, ca_server, and daemon settings
class twitch_core::packages::puppet (
  $puppet_server = 'puppet.prod.us-west2.justin.tv',
  $puppet_ca_server = 'puppet-ca.prod.us-west2.justin.tv',

  $cert_dir = $::osfamily ? {
    'Debian' => '/etc/ssl/certs',
    'RedHat' => '/etc/pki/tls/certs',
    'amazon' => '/etc/pki/tls/certs',
  }
) {
  # If we're building an AMI, we don't want the Puppet agent service to start.
  # If we're running in a non-production environment, we want to even stop the
  # service if it's started.
  if ($::aws_bootstrap == true) {
    $run_style = 'manual'
  } elsif ($::environment != 'production') {
    $run_style = 'external'
  } else {
    $run_style = 'service'
  }

  # This sets the [agent] configurations for `server` and `daemon`
  class { 'puppet::agent':
    puppet_run_style  => $run_style,
    puppet_server     => $puppet_server,
    trusted_node_data => true,
    splay             => true,
  }

  # This sets the [main] configuration for `ca_server`
  # The `puppet::agent` class doesn't expose this configuration, but we can
  # edit the existing file with the same method
  # https://github.com/stephenrjohnson/puppetmodule/issues/1
  ini_setting { 'puppet_main_ca_server':
    ensure  => present,
    path    => '/etc/puppet/puppet.conf',
    section => 'main',
    setting => 'ca_server',
    value   => $puppet_ca_server,
    require => Class['puppet::agent'],
  }

  # Same for rundir apparently
  ini_setting { 'puppet_main_rundir':
    ensure  => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'main',
    setting => 'rundir',
    value   => '/var/run/puppet',
    require => Class['puppet::agent'],
  }

  ini_setting { 'puppet_main_stringify_facts':
    ensure => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'main',
    setting => 'stringify_facts',
    value   => 'false',
    require => Class['puppet::agent'],
  }

  # splaylimit
  ini_setting { 'puppet_agent_splaylimit':
    ensure  => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'agent',
    setting => 'splaylimit',
    value   => '15m',
    require => Class['puppet::agent'],
  }


  # Include Puppet CA root certs for puppet and puppet-ci in each server's PKI
  file { "${cert_dir}/puppet-ca.pem":
    ensure => 'present',
    mode   => '0644',
    owner  => 'root',
    group  => 'root',
    source => "puppet:///modules/${module_name}/ca-certs/puppet-ca.pem",
  }

  file { "${cert_dir}/puppet-ci-ca.pem":
    ensure => 'present',
    mode   => '0644',
    owner  => 'root',
    group  => 'root',
    source => "puppet:///modules/${module_name}/ca-certs/puppet-ci-ca.pem",
  }
}
