# Installs and configures device42-agent, our inventory system for bare metal
class twitch_device42::agent inherits twitch_device42 {

  # create device42 group for other apps that want to read d42 config.
  group { $d42_group:
    ensure => present,
    system => true,
  }

  # create virtualenv
  python::virtualenv { $virtualenv_location:
    ensure  => present,
    owner   => 'root',
    path    => ['/usr/local/bin', '/usr/bin', '/bin'],
  }

  # install twitch_device42
  python::pip { 'twitch_device42':
    ensure     => $twitch_device42_version,
    pkgname    => 'twitch-device42',
    virtualenv => $virtualenv_location,
    require    => Python::Virtualenv[$virtualenv_location],
  }

  # if the agent has never run before, run once
  $state_file = '/etc/hardware/.device42_agent_initialized'
  exec { "${virtualenv_location}/bin/device42-agent ; touch $state_file":
    user    => 'root',
    creates => $state_file,
    require => [
      Python::Pip['twitch_device42'],
    ],
  }

  file { '/etc/twitch_device42/':
    ensure => directory,
    owner => 'root',
    group => $d42_group,
    require => Group[$d42_group],
  }

  file { '/etc/twitch_device42/config':
    ensure  => present,
    owner   => 'root',
    group   => $d42_group,
    mode    => '0640',
    content => template("${module_name}/config.erb"),
    require => [
      File['/etc/twitch_device42/'],
      Group[$d42_group],
    ],
  }

  # run on boot via upstart, after the lshw job
  upstart::job { 'device42_agent':
    ensure         => present,
    description    => 'Device42 Agent',
    start_on       => 'startup',  # should be 'startup and lshw'
    user           => 'root',
    group          => 'root',
    exec           => "${virtualenv_location}/bin/device42-agent",
    service_ensure => 'stopped',  # only runs on boot
    require        => [
      Python::Pip['twitch_device42'],
      File['/etc/twitch_device42/config'],
    ]
  }

  # also run daily via cron
  # use `fqdn_rotate` to randomize the time assignment
  # the same server will always run the job at the same time, but not all
  # servers will run the job at the same time (to avoid overloading device42)
  $cron_hours = fqdn_rotate(range(0, 23))
  $cron_hour = $cron_hours[0]
  $cron_minutes = fqdn_rotate(range(0, 59))
  $cron_minute = $cron_minutes[0]
  cron { 'device42_agent':
    ensure  => present,
    command => "${virtualenv_location}/bin/device42-agent",
    user    => 'root',
    hour    => $cron_hour,
    minute  => $cron_minute,
  }
}
