# This class is not included by the rest of the module. It should be included
# manually if you want to run an ElasticSearch curator instance.
class twitch_elasticsearch::curator (
  $es_host                = 'localhost',
  $es_port                = '9200',
  $snapshot_name          = undef,
  $repository_name        = undef,
  $snapshot_after         = 2,
  $delete_after           = undef,
  $delete_snapshots_after = undef,
) {
  $_connect_string = "--host ${es_host} --port ${es_port}"
  package {'elasticsearch-curator':
    ensure   => '3.5.1',
    provider => 'pip',
    require  => Package['python-pip'],
  }

  if $snapshot_name {
    validate_string($snapshot_name)
    validate_string($repository_name)
    validate_integer($snapshot_after)

    # Run snapshots every day at midnight
    cron {'elasticsearch-snapshots':
      command => join([
        "/usr/local/bin/curator ${_connect_string} snapshot",
        "--prefix ${snapshot_name}-",
        "--repository ${repository_name}",
        "indices --older-than ${snapshot_after} --time-unit days",
        "--timestring '\\%Y.\\%m.\\%d'"
      ], " "),
      user    => 'root',
      hour    => 0,
      minute  => 0,
    }

    file { '/opt/nagios_checks/check_es_snapshots.sh':
      ensure => 'file',
      owner  => 'root',
      group  => 'nagios',
      mode   => '0750',
      source => "puppet:///modules/${module_name}/check_snapshots.sh"
    }

    twitch_nagios::service {'elasticsearch_snapshot':
      is_nrpe => true,
      command => "/opt/nagios_checks/check_es_snapshots.sh ${repository_name} ${snapshot_name}",
      description => 'ElasticSearch Snapshots',
      use     => 'generic-systems-service',
    }

  }

  if $delete_after {
    validate_integer($delete_after)
    cron {'elasticsearch-delete':
      command => join([
        "/usr/local/bin/curator ${_connect_string} delete indices",
        "--older-than ${delete_after} --time-unit days",
        "--timestring '\\%Y.\\%m.\\%d'"
      ], " "),
      user    => 'root',
      hour    => 4,
      minute  => 0,
    }
  }

  # Remove S3 snapshots older than days indicated in delete_snapshots_after var
  if $delete_snapshots_after {
    validate_integer($delete_snapshots_after)
    cron {'elasticsearch-snapshots-pruning':
    command => join([
      "/usr/local/bin/curator ${_connect_string} delete snapshots",
      "--repository ${repository_name}",
      "--older-than ${delete_snapshots_after} --time-unit days"
    ], " "),
    user    => 'root',
    hour    => 1,
    minute  => 0,
    }
  }
}
