class twitch_graphite::install inherits twitch_graphite {
  include twitch_graphite::cleanup

  package { 'buckytools':
    ensure => '0.2.2'
  }
  package { 'go-carbon':
    ensure => '0.8.0'
  }

  # environment where pip installs carbon should be fixed in 1.0.0

  user { $graphite_user:
    ensure  => 'present',
    system  => true,
    gid     => $graphite_group,
    uid     => 978,
    require => Group[$graphite_group]
  }

  # Hack to get the $whisper_mount directory
  # to have the right owner and group
  exec { "fix ${whisper_mount} permissions":
    command     => "chown ${graphite_user}:${graphite_group} ${whisper_mount}",
    refreshonly => true,
  }

  package { 'xfsprogs': } ->
  exec { "format ${mount_device}":
    command => "mkfs.xfs ${mount_device}",
    onlyif  => "[ \"$(file -sL ${mount_device} | grep 'XFS' | wc -l)\" = \"0\" ]"
  } ->
  file{ $whisper_mount:
    ensure  => 'directory',
    group   => $graphite_group,
    owner   => $graphite_user,
    require => User[$graphite_user]
  } ->
  mount { '/mnt':
    ensure => 'absent',
  } ->
  mount { $whisper_mount:
    ensure  => 'mounted',
    atboot  => true,
    fstype  => 'xfs',
    device  => $mount_device,
    options => 'noatime,nodiratime,nobarrier',
    require => File[$whisper_mount],
    notify  => Exec["fix ${whisper_mount} permissions"],
  }

  file{ $whisper_dir:
    group   => $graphite_group,
    owner   => $graphite_user,
    target  => $whisper_mount,
    force   => 'yes',
    require => File[$whisper_mount]
  }

  file { $graphite_dir:
    ensure => 'directory',
    owner  => 'graphite',
    group  => 'graphite',
    mode   => '0755',
  } ->
  file { "${graphite_dir}/pids":
    ensure => 'directory',
    owner  => 'graphite',
    group  => 'graphite',
    mode   => '0755',
  } ->
  file { "${graphite_dir}/conf":
    ensure => 'directory',
    owner  => 'graphite',
    group  => 'graphite',
    mode   => '0755',
  }

  ::python::pip { 'whisper':
    ensure  => $graphite_version,
    require => File[$whisper_dir]
  }

  ::python::pip { 'carbon':
    environment  => "PYTHONPATH=${graphite_dir}/lib",
    ensure       => $graphite_version,
    install_args => '--install-option="--prefix=/opt/graphite" --install-option="--install-lib=/opt/graphite/lib"',
    notify       => Class['twitch_graphite::cleanup'],
  }

  ::python::pip { 'twitch-carbonate':
    require  => ::Python::Pip['carbon']
  }

  class {'rsync::server':
    use_xinetd => false
  }

  rsync::server::module{'whisper':
    path => '/mnt/whisper',
    list => 'yes',
  }

  # manage permissions on dirs created from install
  file { "${graphite_dir}/storage":
    ensure       => 'directory',
    owner        => $graphite_user,
    group        => $graphite_group,
    recurse      => true,
    recurselimit => 1,
    ignore       => 'whisper',
    require      => ::Python::Pip['carbon']
  }

  # manage permissions on dirs created from install
  file { "${graphite_dir}/storage/log":
    ensure       => 'directory',
    owner        => $graphite_user,
    group        => $graphite_group,
    recurse      => true,
    recurselimit => 1,
    ignore       => 'whisper',
    require      => [::Python::Pip['carbon'], File["${graphite_dir}/storage"]]
  }

  group { $graphite_group:
    ensure => 'present',
    system => true,
    gid    => 978,
  }

  file { '/etc/go-carbon':
    ensure => 'directory',
  }

  file { '/var/run/buckytools':
    ensure => 'directory',
    owner  => $graphite_user,
    group  => $graphite_group,
    mode   => '0755',
  }
}
