class twitch_logstash (
  # A list of templates to include in the Logstash configuration
  $config_templates = $twitch_logstash::params::config_templates,
  # The version of Logstash to install
  $version   = $twitch_logstash::params::version,
  $heap_size = $twitch_logstash::params::heap_size,

  # Sometimes we need to add the "logstash" user to the "puppet" group.
  # Specifically, the logstash user needs to read puppet's certs to enable SSL
  # because puppet acts as our PKI.
  $needs_puppet_group = $twitch_logstash::params::needs_puppet_group,

  $template_values = $twitch_logstash::params::template_values,

  $es_template = $twitch_logstash::params::es_template,

  $s3_bucket = $twitch_logstash::params::s3_bucket,
  $s3_folder = $twitch_logstash::params::s3_folder,
) inherits twitch_logstash::params {
  require twitch_core

  validate_hash($config_templates)

  class {'logstash':
    java_install => true,
    manage_repo  => true,
    repo_version => $version,
  }

  if $needs_puppet_group {
    # Add the logstash user to the puppet group
    exec {'logstash puppet group membership':
      unless => "grep -q 'puppet\\S*logstash' /etc/group",
      command => 'usermod -aG puppet logstash',
      require => Package['logstash'],
      before  => Service['logstash'],
    }
  }

  if $es_template {
    file {"/etc/logstash/${es_template}":
      ensure => 'present',
      source => "puppet:///modules/${module_name}/${es_template}",
      require => Class['logstash'],
    }
  }

  if has_key($config_templates, 'relp_input') {
    exec { 'install-relp-plugin':
      command => "${::logstash::plugin} install logstash-input-relp",
      onlyif  => "test ! -d ${::logstash::installpath}/vendor/bundle/jruby/*/gems/logstash-input-relp*",
      require => Package['logstash'],
      before  => Service['logstash'],
    }
  }

  logstash::plugin { 'logstash-filter-prune':
  }

  file {'/etc/default/logstash':
    ensure  => 'present',
    content => template("${module_name}/defaults.erb"),
    require => Class['logstash'],
  }

  create_resources('logstash::configfile', $config_templates)

  # HACK: Logstash's RabbitMQ output plugin only allows "fanout", "direct", and
  # "topic" exchanges. We need "x-consistent-hash" for our infrastructure to
  # work, so we have to do.... this. It's not pretty, and I apologize to you
  # personally, whoever is reading this. I'm sorry.
  if has_key($config_templates, 'rabbit_output') {
    file { '/opt/logstash/vendor/bundle/jruby/1.9/gems/logstash-output-rabbitmq-1.1.1-java/lib/logstash/outputs/rabbitmq.rb':
      owner => 'logstash',
      group => 'logstash',
      mode  => '0644',
      source  => "puppet:///modules/${module_name}/rabbitmq-replacement.rb",
      require => Package['logstash'],
      before  => Service['logstash'],
    }
  }

  # Ticket: SYS-8978
  file {'/opt/logstash/bin/logstash.lib.sh':
    ensure  => 'present',
    owner => 'logstash',
    group => 'logstash',
    mode  => '0775',
    source  => "puppet:///modules/${module_name}/logstash.lib.sh",
    require => Class['logstash'],
  }

  include twitch_logstash::monitoring
  include twitch_logstash::logrotate
}
