# Installs and configures mgdb-agent, our inventory system
# https://git-aws.internal.justin.tv/video/twitch_mgdbagent
class twitch_mgdb::agent inherits twitch_mgdb::params {
  require twitch_lshw

# User and group are in LDAP, but ensure on the system as well.
  user { $twitch_mgdb::params::agent_user:
    ensure     => present,
    uid        => $twitch_mgdb::params::uid,
    gid        => $twitch_mgdb::params::gid,
    home       => $twitch_mgdb::params::agent_home,
    password   => '!!',
    managehome => false,
    forcelocal => true,
    shell      => '/bin/bash',
    require    => Group[$twitch_mgdb::params::agent_group],
  }

  group { $twitch_mgdb::params::agent_group:
    ensure     => present,
    gid        => $twitch_mgdb::params::gid,
    forcelocal => true,
  }

  file { "${twitch_mgdb::params::agent_home}/logs":
    ensure  => directory,
    owner   => $twitch_mgdb::params::agent_user,
    group   => $twitch_mgdb::params::agent_group,
    require => [
      User[$twitch_mgdb::params::agent_user],
      File[$twitch_mgdb::params::agent_home]
    ],
  }

  # Ensure home dir exists, some wierdness with ldap/local users
  file { $twitch_mgdb::params::agent_home:
    ensure  => directory,
    owner   => $twitch_mgdb::params::agent_user,
    group   => $twitch_mgdb::params::agent_group,
    require => User[$twitch_mgdb::params::agent_user],
  }

  # Make sure this user has local pip settings
  file { "${twitch_mgdb::params::agent_home}/.pip":
    ensure  => directory,
    owner   => $twitch_mgdb::params::agent_user,
    group   => $twitch_mgdb::params::agent_group,
    require => [
      User[$twitch_mgdb::params::agent_user],
      File[$twitch_mgdb::params::agent_home]
    ],
  }

  file { "${twitch_mgdb::params::agent_home}/.pip/pip.conf":
    ensure  => present,
    owner   => $twitch_mgdb::params::agent_user,
    group   => $twitch_mgdb::params::agent_group,
    source  => 'puppet:///modules/twitch_mgdb/pip.conf',
    require => [
      User[$twitch_mgdb::params::agent_user],
      File["${twitch_mgdb::params::agent_home}/.pip"]
    ],
  }

  # Run mgdb-agent once per hour (at a random time) to keep inventory up-to-date
  # Use 3400 seconds to avoid potential overlap of mgdb-agent runs
  $agent_args = "--in-service yes --verbose --wait ${twitch_mgdb::params::wait_time}"
  cron { 'mgdbagent':
    ensure      => present,
    command     => ". ${twitch_mgdb::params::virtualenv_location}/bin/activate > /dev/null 2>&1 && pip -q install -q --upgrade twitch_mgdbagent > /dev/null 2>&1; pgrep mgdb-agent > /dev/null 2>&1 || mgdb-agent ${agent_args} >> ${twitch_mgdb::params::agent_home}/logs/mgdb-agent.log 2>&1 ",
    environment => "PATH=${twitch_mgdb::params::virtualenv_location}/bin:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin",
    user        => $twitch_mgdb::params::agent_user,
    hour        => '*',
    minute      => fqdn_rand(60),
    require     => [
      User[$twitch_mgdb::params::agent_user],
      Sudo::Conf['ttv-mgdbagent']
    ],
  }

  sudo::conf { 'ttv-mgdbagent':
    priority => 99,
    source   => 'puppet:///modules/twitch_mgdb/sudoers',
  }

  logrotate::rule { 'mgdbagent':
    ensure       => present,
    path         => "${twitch_mgdb::params::agent_home}/logs/mgdb-agent.log",
    missingok    => true,
    rotate_every => 'day',
    rotate       => 7,
    ifempty      => false,
    compress     => true,
    require      => [
      File["${twitch_mgdb::params::agent_home}/logs"],
      User[$twitch_mgdb::params::agent_user]
    ],
  }

  # Make sure old job is gone
  cron { 'mgdb-agent':
    ensure => absent,
    user   => 'root',
  }
}
