#!/usr/bin/env python

import logging
from logging.handlers import SysLogHandler
import os
import socket
import yaml
import datetime
import re
from jira import JIRA

PROJECT_ID = 14604
NAGIOS_CMD_LOCATION = '/var/nagios/rw/nagios.cmd'
NAGIOS_LIVE_SOCK = '/var/nagios/rw/live'

os.environ['https_proxy'] = 'http://proxy.internal.justin.tv:9797'
os.environ['http_proxy'] = 'http://proxy.internal.justin.tv:9797'

LOGGER = logging.getLogger('syslog')
formatter = logging.Formatter("jira_downtime [%(process)s]: %(levelname)s - %(message)s")
handler = SysLogHandler('/dev/log', facility=logging.handlers.SysLogHandler.LOG_LOCAL3)
handler.setFormatter(formatter)
LOGGER.addHandler(handler)

def get_downtimes():
    """This hits the livestatus lql call for getting all current downtimes"""
    try:
        #build the LQL query, to find columns, query like this check 1st row of a blank query
        message = 'GET downtimes\n'
        message += 'Columns: host_name end_time comment service_display_name\n'
        message += 'OutputFormat: json\n\n'

        nagios_socket_connection = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        nagios_socket_connection.connect(NAGIOS_LIVE_SOCK)
        #send the message, shutdown the socket's messaging. Need to close?
        nagios_socket_connection.send(message)
        # Now read the answer
        answer = nagios_socket_connection.recv(100000000)
        nagios_socket_connection.shutdown(socket.SHUT_WR)
        yaml_answer = yaml.load(answer)

        nagios_socket_connection.close()
    except Exception as e:
        LOGGER.warn("LiveStatus Socket Broke:\n" + str(e))

    #spit the answer back in YAML
    return yaml_answer

def check_downtimes(jira_connection):
    """take the yaml and investigate each one for a ticket number & when it expires"""
    ticket_regex = re.compile('(?i)(^VIDOPS-[0-9]+|^DC-[0-9]+|^NET-[0-9]+)')
    for downtime in get_downtimes():
        downtime_comment = downtime[2]
        downtime_ticket_num = ticket_regex.findall(downtime_comment)
        downtime_date = datetime.datetime.fromtimestamp(int(downtime[1]))
        current_date = datetime.datetime.now()
        #check list is not-empty & that the current date + 1 day is greater than when downtime ends

        if downtime_ticket_num and (current_date + datetime.timedelta(1) > downtime_date):
            jira_ticket_number = check_jira(jira_connection, downtime_ticket_num[0])
            if jira_ticket_number:
                new_downtime(downtime_ticket_num[0], downtime[0], downtime[3])

def check_jira(jira_connection, id_from_comment):
    """Connect to JIRA and look for tickets with the same key found in comment"""
    print id_from_comment
    try:
        ticket_search = jira_connection.search_issues('id = "' +    \
            id_from_comment.upper() + '" AND project = ' + str(PROJECT_ID)+ \
            ' AND status != "Closed"', maxResults=1)
        if ticket_search[0]:
            ticket_result = ticket_search[0].key
        else:
            ticket_result = None
    except Exception as e:
        LOGGER.warn("TICKET SEARCH FAILURE:\n" + str(e))
        ticket_result = None
    LOGGER.warn("TICKET RESULT:\n" + str(ticket_result))
    return ticket_result

def new_downtime(jira_ticket_number, host_name, service_name):
    current_time = datetime.datetime.now()
    end_time = current_time + datetime.timedelta(7)

    #nagios likes epoch time, so let's convert-
    epoch_end_time = end_time.strftime('%s')
    epoch_current_time = current_time.strftime('%s')

    #if there is a string for "service" then downtime as a service, else host.
    if service_name:
        downtime_string = "COMMAND [{0}] SCHEDULE_SVC_DOWNTIME;{1};{2};{3};{4};1;0;7200;auto-nagios;{5}\n".format(epoch_current_time, host_name, service_name, epoch_current_time, epoch_end_time, jira_ticket_number)
    else:
        downtime_string = "COMMAND [{0}] SCHEDULE_HOST_DOWNTIME;{1};{2};{3};1;0;7200;auto-nagios;{4}\n".format(epoch_current_time, host_name, epoch_current_time, epoch_end_time, jira_ticket_number)

    #this opens up the nagios.cmd socket and writes the command to it, 'with:' closes when it's done.
    try:
        with open(NAGIOS_CMD_LOCATION, 'w') as nagios_cmd:
            nagios_cmd.writelines(downtime_string)
    except:
        LOGGER.warn("Cannot write to Nagios.cmd!!")

def connect_to_jira():
    """Connects to JIRA as user video-nagios"""

    jira_options = {'server': "https://twitchtv.atlassian.net"}
    jira_cert_location = '/etc/nagios/static/eventhandlers/jira.auth'
    try:
        with open(jira_cert_location,'r') as jira_cert:
            jira_password = jira_cert.readline()
            #avoid the new line char
            jira_connection = JIRA(options=jira_options, basic_auth=('video-nagios', jira_password), validate=True)
    except Exception as e:
        LOGGER.warn("Cannot connect to JIRA" + str(e))
        exit()
    return jira_connection

if __name__ == '__main__':

    check_downtimes(connect_to_jira())
