# Configures the base Nagios system
class twitch_nagios::server::configs inherits twitch_nagios::server {
  # Define the service here to avoid cycles in the anchor pattern
  service { "nagios":
    ensure     => running,
    enable     => true,
    hasstatus  => true,
    hasrestart => true,
  }

  service { "nsca":
    ensure     => running,
    enable     => true,
    hasstatus  => false,
    hasrestart => true,
  }

  # Make sure ownership on these files is right.
  file { [
    $twitch_nagios::server::host_config_file,
    $twitch_nagios::server::service_config_file,
    $twitch_nagios::server::hostgroup_config_file,
    $twitch_nagios::server::servicegroup_config_file,
    ]:
    ensure => 'present',
    owner  => 'nagios',
    group  => 'nagios',
    mode   => '0664',
    notify => Service['nagios'],
  }

  # Populate /etc/nagios/objects from twitch_nagios/files/server/objects
  file { "/etc/nagios/objects":
    ensure  => directory,
    owner   => "nagios",
    group   => "nagios",
    mode    => "0755",
    source  => "puppet:///modules/${module_name}/server/objects",
    recurse => true,
    purge   => true,
  }

  file { '/var/lib/nagios/checks':
    ensure => directory,
    owner  => 'nagios',
    group  => 'nagios',
    source  => "puppet:///modules/${module_name}/server/common_checks",
    recurse => true,
  }

  file { '/etc/init.d/nagios':
    ensure => present,
    owner  => 'root',
    group  => 'root',
    mode    => '0755',
    content => template("${module_name}/server/nagios-init.erb"),
  }

  # nagios_{host,service,hostgroup,servicegroup} resources only support purging
  # if they're collected into the default locations. In collectors.pp we set a
  # "target =>" for each of our resource types, so since we want to use purging
  # (SURPRISE) we need a hack. We symlink the default target for these files to
  # the places that we actually put them so that puppet knows how to purge them.

  $_default_host_file = '/etc/nagios/nagios_host.cfg'
  if $twitch_nagios::server::host_config_file != $_default_host_file {
    file { $_default_host_file:
      ensure => 'link',
      target => $twitch_nagios::server::host_config_file,
      owner  => 'nagios',
      group  => 'nagios',
    }
  }

  $_default_service_file = '/etc/nagios/nagios_service.cfg'
  if $twitch_nagios::server::service_config_file != $_default_service_file {
    file { $_default_service_file:
      ensure => 'link',
      target => $twitch_nagios::server::service_config_file,
      owner  => 'nagios',
      group  => 'nagios',
    }
  }

  $_default_hostgroup_file = '/etc/nagios/nagios_hostgroup.cfg'
  if $twitch_nagios::server::hostgroup_config_file != $_default_hostgroup_file {
    file { $_default_hostgroup_file:
      ensure => 'link',
      target => $twitch_nagios::server::hostgroup_config_file,
      owner  => 'nagios',
      group  => 'nagios',
    }
  }

  $_default_servicegroup_file = '/etc/nagios/nagios_servicegroup.cfg'
  if $twitch_nagios::server::servicegroup_config_file != $_default_servicegroup_file {
    file { $_default_servicegroup_file:
      ensure => 'link',
      target => $twitch_nagios::server::servicegroup_config_file,
      owner  => 'nagios',
      group  => 'nagios',
    }
  }
}
