define twitch_nagios::service (
  $command,
  $description,

  $ensure  = 'present',
  $is_nrpe = false,
  $nrpe_sudo = false,
  $use = 'generic-service',
  $notes_url = undef,
  $extra_servicegroups = []
) {
  validate_string($command)
  validate_string($description)
  validate_re($ensure, [ '^present$', '^absent$'])
  validate_bool($is_nrpe)
  validate_bool($nrpe_sudo)
  validate_string($use)
  validate_array($extra_servicegroups)

  if $is_nrpe {
    $_nrpe_name = "check_${name}"
    $_check_command = "check_nrpe!${_nrpe_name}"

    ::nrpe::command { $_nrpe_name:
      ensure  => $ensure,
      command => $command,
      sudo    => $nrpe_sudo,
    }

  } else {
    $_check_command = $command
  }

  $_git_server = hiera('twitch_git_server')
  # We have to look this up with hiera rather than with scoping, because
  # twitch_nagios::server isn't included on most servers.
  $_nagios_repo = hiera('twitch_nagios::server::config_repo', undef)

  if $notes_url {
    $_notes_url = $notes_url
  } else {
    $_notes_url = $_nagios_repo ? {
      undef   => undef,
      default => "https://${_git_server}/${_nagios_repo}/wiki/Alerts_${name}"
    }
  }

  # Default servicegroups are, for example, "cpu", and "video-edge cpu"
  $_default_servicegroups = [
    $name,
    "${::twitch_role} ${name}",
  ]

  $all_servicegroups = unique(concat($_default_servicegroups, $extra_servicegroups))

  @@nagios_service { "${::fqdn}_${name}":
    ensure              => $ensure,
    check_command       => $_check_command,
    use                 => $use,
    host_name           => $::fqdn,
    service_description => $description,
    tag                 => "team_${::twitch_team}_${::twitch_environment}",
    servicegroups       => join($all_servicegroups, ','),
    notes_url           => $_notes_url,
  }

  # Also register a servicegroup automagically
  # Turns ['a', 'b'] into ['a#fqdn', 'b#fqdn'], see twitch_nagios::servicegroup
  # docs for details.
  $servicegroup_labels = suffix($all_servicegroups, "#${::fqdn}")
  @@twitch_nagios::servicegroup { $servicegroup_labels:
    export => true,
    tag    => "team_${::twitch_team}_${::twitch_environment}",
  }
}
