# Checks out a repository of puppet code to be "puppet apply"d.
#
# Referred to as "application puppet", whereas this repo, puppet-ci, is more
# of a "systems puppet".
#
# Parameters:
#  $destination - The folder to place the repo in, defaults to /etc/puppet-local
#  $git_server  - Git server to pull from. Defaults to hiera('twitch_git_server')
#  $repo        - REQUIRED. The repository to check out.
#  $repo_autoupdate - Whether to let puppet keep the repo up to date. Defaults to false.
#  $key         - REQUIRED for now. SSH key to place in /var/lib/puppet/.ssh

class twitch_puppet_apply (
  $destination      = $twitch_puppet_apply::params::destination,

  $git_server       = hiera('twitch_git_server'),
  $repo             = $twitch_puppet_apply::params::repo,
  $repo_autoupdate  = $twitch_puppet_apply::params::repo_autoupdate,
  $repo_revision    = $twitch_puppet_apply::params::repo_revision,

  $key              = $twitch_puppet_apply::params::key,

  $auto_apply       = $twitch_puppet_apply::params::auto_apply,
) inherits twitch_puppet_apply::params {

  require twitch_core

  validate_string($destination)
  validate_string($git_server)
  validate_string($repo)
  validate_bool($repo_autoupdate)
  validate_string($repo_revision)
  validate_string($key)
  validate_bool($auto_apply)

  # This package is required to send reports to puppetdb.
  package {'puppetdb-terminus':
    ensure => 'installed',
  }

  # TODO: SOON! Move this key into Sandstorm. Blocked on baremetal sandstorm.
  file { '/var/lib/puppet/.ssh':
    ensure  => 'directory',
    source  => "puppet:///modules/${module_name}/keys/${key}",
    owner   => 'puppet',
    group   => 'puppet',
    mode    => '0700',
    recurse => 'remote',
    purge   => false,
  }

  file { $destination:
    ensure => 'directory',
    owner  => 'puppet',
    group  => 'puppet',
    mode   => '0755',
  }

  $_repo_url = "${git_server}:${repo}"

  if $repo_autoupdate {
    $_vcsrepo_ensure = 'latest'
  } else {
    $_vcsrepo_ensure = 'present'
  }

  $_puppet_apply_cmd = "puppet apply --verbose --show_diff --modulepath ${destination}/modules --hiera_config ${destination}/hiera.yaml ${destination}/manifests/site.pp"

  $_vcsrepo_notify = $auto_apply ? {
    true    => Exec[$_puppet_apply_cmd],
    default => undef,
  }

  vcsrepo { $destination:
    ensure   => $_vcsrepo_ensure,
    revision => $repo_revision,
    provider => 'git',
    source   => "git@${_repo_url}.git",
    user     => 'puppet',
    owner    => 'puppet',
    group    => 'puppet',
    require  => [
      File['/var/lib/puppet/.ssh'],
      File[$destination],
    ],
    notify  => $_vcsrepo_notify,
  }

  exec { $_puppet_apply_cmd:
    logoutput   => true,
    refreshonly => true,
  }

}
