# Installs and configures Puppet master server
class twitch_puppet_master (
  $certificate_authority = false,
  $storeconfigs          = true,
  $enable_webhook        = true,
  $version        = 'present',
) {
  require twitch_core
  include twitch_puppet_master::logging

  if $enable_webhook {
    include twitch_puppet_master::webhook_receiver
  }

  $ssldir = '/var/lib/puppet/ssl'

  file { '/etc/default/puppetmaster':
    ensure => 'file',
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => "puppet:///modules/${module_name}/defaults",
  }


  # If we're a CA we need an extra alt name in our cert
  if $certificate_authority {
    $_dns_alt_names = [
      $::twitch_core::packages::puppet::puppet_server,
      $::twitch_core::packages::puppet::puppet_ca_server,
    ]
  } else {
    # If we aren't a CA, not only do we only need one alt name,
    $_dns_alt_names = [
      $::twitch_core::packages::puppet::puppet_server,
    ]

    # but we also need to add some symlinks for certs. The puppet::master module
    # assumes that the puppet certs will be in the location they're in on CA
    # servers, while non-ca servers actually keep them in different places.
    # Instead of modifying the module (bad!), we'll just add some symlinks to
    # trick it into playing nice.
    file { "${ssldir}/ca/ca_crt.pem":
      ensure => 'link',
      target => "${ssldir}/certs/ca.pem",
    }
    file { "${ssldir}/ca/ca_crl.pem":
      ensure => 'link',
      target => "${ssldir}/crl.pem",
    }
  }

  file { '/etc/puppet/auth.conf':
    ensure  => 'file',
    owner   => 'puppet',
    group   => 'puppet',
    mode    => '0644',
    content => template("${module_name}/auth.conf.erb"),
    notify  => Service['puppetmaster'],
  }

  class { 'puppet::master':
    autosign                 => true,
    dns_alt_names            => $_dns_alt_names,
    puppet_ssldir            => $ssldir,
    puppetdb_startup_timeout => 10,
    reports                  => 'store, puppetdb',
    storeconfigs             => $storeconfigs,
    storeconfigs_dbserver    => 'puppetdb-2a748adc.prod.us-west2.justin.tv',
    environments             => 'directory',
    version                  => $version,
    require                  => Class['twitch_core::repositories::apt'],
  }

  # Allow the CA to be selectively enabled/disabled
  # https://github.com/stephenrjohnson/puppetmodule/issues/1

  # Turn $certificate_authority into a string for the ini_setting
  $_certificate_authority = $certificate_authority ? {
    true    => 'true',
    default => 'false',
  }

  ini_setting { 'puppet_master_ca':
    ensure  => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'master',
    setting => 'ca',
    value   => $_certificate_authority,
    require => Class['puppet::master'],
  }

  # Recommended by the Puppet docs for performance:
  # https://docs.puppetlabs.com/guides/passenger.html#configure-puppet
  ini_setting { 'puppet_master_cache_features':
    ensure  => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'master',
    setting => 'always_cache_features',
    value   => 'true',
    require => Class['puppet::master'],
  }

  # enable trusted node data by default (enables facts hash use)
  ini_setting { 'puppet_master_trusted_node_data':
    ensure  => 'present',
    path    => '/etc/puppet/puppet.conf',
    section => 'master',
    setting => 'trusted_node_data',
    value   => 'true',
    require => Class['puppet::master'],
  }

  # Allows querying puppetdb (e.g. for using nodequery)
  # This gem is required for the puppetdbquery module
  package { 'ruby-puppetdb':
    ensure   => '1.6.1', #pinned so it doesnt install puppet 4
    provider => 'gem'
  }

  # Removes report files/directories after 1 week
  tidy { '/var/lib/puppet/reports':
    age     => '1w',
    recurse => true,
  }

}
