require 'shellwords'
require 'puppet/provider'
require 'puppet/util/execution'

Puppet::Type.type(:sandstorm_resource).provide :commands do
  include Puppet::Util::Execution
  desc "Executes the commands required to maintain the status of a sandstorm resource."

  confine :feature => :posix
  defaultfor :feature => :posix

  # checkexe and run are Copypasta'd from puppet's provider/exec/posix because
  # I don't know how to patch it in the correct way Kappa
  # Verify that we have the executable
  def checkexe(command)
    exe = extractexe(command)

    if File.expand_path(exe) == exe
      if !Puppet::FileSystem.exist?(exe)
        raise ArgumentError, "Could not find command '#{exe}'"
      elsif !File.file?(exe)
        raise ArgumentError, "'#{exe}' is a #{File.ftype(exe)}, not a file"
      elsif !File.executable?(exe)
        raise ArgumentError, "'#{exe}' is not executable"
      end
      return
    end

    # 'which' will only return the command if it's executable, so we can't
    # distinguish not found from not executable
    raise ArgumentError, "Could not find command '#{exe}'"
  end

  def extractexe(command)
    if command.is_a? Array
      command.first
    elsif match = /^"([^"]+)"|^'([^']+)'/.match(command)
      # extract whichever of the two sides matched the content.
      match[1] or match[2]
    else
      command.split(/ /)[0]
    end
  end

  def run(command, check = false)
    output = nil
    status = nil
    dir = nil

    checkexe(command)

    if dir = resource[:cwd]
      unless File.directory?(dir)
        if check
          dir = nil
        else
          self.fail "Working directory '#{dir}' does not exist"
        end
      end
    end

    dir ||= Dir.pwd

    debug "Executing#{check ? " check": ""} '#{command}'"
    begin
      # Do our chdir
      Dir.chdir(dir) do
        environment = {}

        environment[:PATH] = resource[:path].join(File::PATH_SEPARATOR) if resource[:path]

        if envlist = resource[:environment]
          envlist = [envlist] unless envlist.is_a? Array
          envlist.each do |setting|
            if setting =~ /^(\w+)=((.|\n)+)$/
              env_name = $1
              value = $2
              if environment.include?(env_name) || environment.include?(env_name.to_sym)
                warning "Overriding environment setting '#{env_name}' with '#{value}'"
              end
              environment[env_name] = value
            else
              warning "Cannot understand environment setting #{setting.inspect}"
            end
          end
        end

        # Ruby 2.1 and later interrupt execution in a way that bypasses error
        # handling by default. Passing Timeout::Error causes an exception to be
        # raised that can be rescued inside of the block by cleanup routines.
        #
        # This is backwards compatible all the way to Ruby 1.8.7.
        Timeout::timeout(resource[:timeout], Timeout::Error) do
          # note that we are passing "false" for the "override_locale" parameter, which ensures that the user's
          # default/system locale will be respected.  Callers may override this behavior by setting locale-related
          # environment variables (LANG, LC_ALL, etc.) in their 'environment' configuration.
          output = Puppet::Util::Execution.execute(command, :failonfail => false, :combine => true,
                                  :uid => resource[:user], :gid => resource[:group],
                                  :override_locale => false,
                                  :custom_environment => environment)
        end
        # The shell returns 127 if the command is missing.
        if output.exitstatus == 127
          raise ArgumentError, output
        end

      end
    rescue Errno::ENOENT => detail
      self.fail Puppet::Error, detail.to_s, detail
    end

    return output
  end

  def run_with_secret(cmd)
    secret_cmd = interpolate_secret(cmd)
    results = run(secret_cmd)
    results.exitstatus == 0
  end

  def interpolate_secret(str)
    secrets_file = @resource[:secret_file]
    err "Secrets file #{secrets_file} not found. Is Sandstorm running?" unless File.file?(secrets_file)
    secret = File.open(secrets_file, 'r') { |f| f.read }
    secret.rstrip! # The secrets file has a newline at the end. No gucci.
    # Ruby doesn't support %q, so replace it here. We want the %q to show up
    # in the bash stuff that the puppet manifests create.
    str.gsub!("%q", "%s")
    str % Shellwords.escape(secret)
  end

  def create
    run_with_secret(@resource[:create_command])
  end

  def destroy
    results = run(@resource[:destroy_command])
    results.exitstatus == 0
  end

  def exists?
    results = run(@resource[:exists_command])
    results.exitstatus == 0
  end

  def insync?(is)
    run_with_secret(resource[:in_sync_command])
  end

  def update_secret
    run_with_secret(resource[:update_command])
  end

  def in_sync_command
    nil
  end

end
