Puppet::Type.newtype(:sandstorm_resource) do
  desc %q{Native type for managing non-file Sandstorm resources.

    Usage of this type is intended for resources such as database users which
    don't correspond to plaintext files on the server's filesystem.

    Normally you won't directly use a sandstorm_resource in your service
    manifests. We've also supplied a twitch_sandstorm_agent::resource class
    which wraps some of the basic boilerplate required to use this resource
    effectively. See that class for information on how to use it.

    "%s" will be replaced with the value of the secret in question.

    "user" and "group" are the uid/gid to run commands as.

    Example:

        sandstorm_resource { "rabbitmq_user_${username}":
          ensure          => 'present',
          secret_file     => $_secret_file,
          exists_command  => "/usr/sbin/rabbitmqctl -q list_users | grep '^${username}'",
          create_command  => "/usr/sbin/rabbitmqctl add_user ${username} %s",
          destroy_command => "/usr/sbin/rabbitmqctl delete_user ${username}",
          in_sync_command => "/usr/sbin/rabbitmqctl authenticate_user ${username} %s",
          update_command  => "/usr/sbin/rabbitmqctl change_password ${username}
          require         => Twitch_sandstorm_agent::Template["resource-${name}"],
        }
  }

  ensurable

  newparam(:name) do
    desc %q{The name of the resource to manage. This won't be used directly by
      the resource commands, but it's still important that it's unique, as it
      will be the name of the sandstorm template.
    }
  end

  newparam(:user) do
    desc "The user to run the various commands as."
    defaultto "root"

    # Validation stolen from Puppet's native Exec type
    validate do |user|
      if Puppet.features.microsoft_windows?
        self.fail "Unable to execute commands as other users on Windows"
      end
    end
  end

  newparam(:group) do
    desc "The group to run various commands as."
    defaultto "root"
  end

  # Cwd, path, environment taken directly from Puppet::Type::Exec because why not
  newparam(:cwd, :parent => Puppet::Parameter::Path) do
    desc "The directory from which to run the command.  If
      this directory does not exist, the command will fail."
  end

  newparam(:path) do
    desc "The search path used for command execution.
      Commands must be fully qualified if no path is specified.  Paths
      can be specified as an array or as a '#{File::PATH_SEPARATOR}' separated list."

    # Support both arrays and colon-separated fields.
    def value=(*values)
      @value = values.flatten.collect { |val|
        val.split(File::PATH_SEPARATOR)
      }.flatten
    end
  end

  newparam(:environment) do
    desc "Any additional environment variables you want to set for a
      command.  Note that if you use this to set PATH, it will override
      the `path` attribute.  Multiple environment variables should be
      specified as an array."

    validate do |values|
      values = [values] unless values.is_a? Array
      values.each do |value|
        unless value =~ /\w+=/
          raise ArgumentError, "Invalid environment setting '#{value}'"
        end
      end
    end
  end

  newparam(:timeout) do
    desc "The maximum time the command should take.  If the command takes
      longer than the timeout, the command is considered to have failed
      and will be stopped. The timeout is specified in seconds. The default
      timeout is 300 seconds and you can set it to 0 to disable the timeout."

    munge do |value|
      value = value.shift if value.is_a?(Array)
      begin
        value = Float(value)
      rescue ArgumentError
        raise ArgumentError, "The timeout must be a number.", $!.backtrace
      end
      [value, 0.0].max
    end

    defaultto 300
  end

  newparam(:secret_file) do
    desc "The file that Sandstorm placed which contains _only_ the plaintext secret"
    validate do |s|
      unless s
        self.fail "secret_file is required"
      end
    end
  end

  newparam(:exists_command) do
    desc %q{Command to run to check if the resource exists at all. If this
      command returns zero, the resource is considered to exist.
    }
  end

  newparam(:create_command) do
    desc %q{Command to run to create the resource, if it doesn't already exist
      and ensure is set to 'present'.
    }
  end

  newparam(:destroy_command) do
    desc %q{Command to run to destroy this resource, if it exists and ensure is
      set to 'absent'.
    }
  end

  # Since this resource is slightly different than other resources, in that it
  # doesn't map a property directly to a value on the target system, we're
  # choosing this command to mark as a property. It's the closest thing we can
  # give to puppet as "This is the value that could change", and only
  # properties (not parameters) can have an 'insync?' method.
  newproperty(:in_sync_command) do
    desc %q{Command to run to check if the resource's secret is in sync. This
      command will be run when the resource exists. An exit status of zero
      indicates that the secret is in sync, non-zero indicates out of sync.
    }

    def insync?(is)
      provider.insync? is
    end
    def set(value)
      provider.update_secret
    end
    def change_to_s(current, desired)
      "secret has been updated"
    end
  end

  newparam(:update_command) do
    desc %q{Command to run to update the resource's secret. This command will be
      run when the in_sync_command indicates that the secret is out of sync.

      IMPORTANT: This command will also be passed to Sandstorm as the command it
      should run when a secret updates.
    }
  end

  # Autorequire statement stolen from Puppet's native Exec type
  autorequire(:user) do
    # Autorequire users if they are specified by name
    if user = self[:user] and user !~ /^\d+$/
      user
    end
  end

  autorequire(:exec) { "sandstorm run #{name}" }
end
