# define: twitch_sandstorm_agent::template
#
# Creates a file with values from our secrets management system interpolated.
#
# Parameters:
#   [*ensure*]          - "present", "absent", state of managed resources
#   [*destination*]     - The path where the final file should be placed
#   [*restart_command*] - Shell command to run which restarts the service that
#                         depends on the destination file.
#   [*contents*]        - String, the contents of the template file to be used.
#                         Use either `contents` OR `key`, not both.
#   [*key*]             - String, the secret key that should be placed into the
#                         destination file. Use this when a file should only
#                         contain a single secret and nothing else.
#   [*delete_dest*]     - Boolean, whether or not to delete the destination
#                         file when ensure = absent on this resource. Has no
#                         effect when ensure = present. Defaults to false.
#
# Usage:
#  twitch_sandstorm_agent::template { 'wildcard-jtv-cert':
#    destination => '/tmp/test-cert',
#    contents    => template("${module_name}/cert.erb"),
#  }
#
# twitch_sandstorm_agent::template { 'wildcard-jtv-cert':
#   destination => '/tmp/test-cert',
#   key         => 'WILDCARD_INTERNAL_JUSTIN_TV_CERT',
# }
#
# twitch_sandstorm_agent::template { 'wildcard-jtv-cert':
#   destination => '/tmp/test-cert',
#   source         => '/opt/twitch/web/current/config/settings/staging_tmpl.json',
# }
#
# twitch_sandstorm_agent::template { 'wildcard-jtv-cert':
#   destination => '/tmp/test-cert',
#   file  => "puppet:///modules/${module_name}/pgpass",
# }
#
#  twitch_sandstorm_agent::template { 'wildcard-jtv-cert':
#    destination => '/tmp/test-cert',
#    owner       => 'custom_user',
#    group       => 'custom_group',
#    mode        => '0600'        # only 'custom_user' can read/write the file
#    template    => "${module_name}/cert.erb",
#  }
#

define twitch_sandstorm_agent::template (
  $destination = '',
  $allow_empty = false,
  $restart_command = '',
  $contents = undef,
  $key = undef,
  $template = undef,
  $ensure = 'present',
  $owner = 'root',
  $group = 'root',
  $file = undef,
  $trim_newline = false,
  $source = undef,
  $mode  = '0644',
  $delete_dest = false,
) {

  include twitch_sandstorm_agent

  if $source {
    $template_source = $source
    $config_require = undef
  } else {
    $template_source = "${twitch_sandstorm_agent::params::templates_dir}/${name}"
    $config_require = File[$template_source]
  }

  $_exec_name = "sandstorm run ${name}"
  $_exec_cmd = "${twitch_sandstorm_agent::sandstorm_run_cmd} --template ${template_source}"
  exec { $_exec_name:
    command     => $_exec_cmd,
    cwd         => '/root',
    environment => ['HOME=/root'],
    refreshonly => true,
    subscribe   => [
      Package[$twitch_sandstorm_agent::package_name],
    ],
    notify      => [
      Service['monit-sandstorm-agent'],
    ],
  }

  if $source == undef {
    # If a source file is specified manually, don't manage it in puppet

    if ($contents == undef) and ($key == undef) and ($template == undef) and ($file == undef) and !$allow_empty {
      fail('twitch_sandstorm_agent::template requires either $contents or $key or $template or $file, none were provided. Please set $allow_empty if you really want this to be blank')
    }

    $options = [$contents, $key, $template, $file]

    if count($options) > 1 {
      fail('twitch_sandstorm_agent::template requires either $contents, $key, $key, $template, or $file, too many options were provided')
    }

    if $file {
      file { $template_source:
        ensure => $ensure,
        source => $file,
        notify => Exec[$_exec_name],
      }
    } else {
      if $contents {
        $file_content = $contents
      } elsif $template {
        $file_content = template($template)
      } elsif $key {
        if $trim_newline {
          $file_content = "{{ key \"${key}\" }}\n{{- /*suppress trailing new line*/ -}}"
        } else{
          $file_content = "{{ key \"${key}\" }}"
        }
      } elsif $allow_empty {
          $file_content = ""
      }

      file { $template_source:
        ensure  => $ensure,
        content => $file_content,
        notify  => Exec[$_exec_name],
      }
    }
  }

  file { "${twitch_sandstorm_agent::confd_dir}/10-${name}.conf":
    ensure  => $ensure,
    content => template("${module_name}/template.erb"),
    notify  => [
      Exec[$_exec_name],
    ],
    require => $config_require,
  }

  if $ensure == 'absent' and $delete_dest {
    file { $destination:
      ensure => 'absent',
      # Make sure the file isn't bucketed by puppet
      backup => false,
    }
  }
}
