# "Abandon all hope, ye who enter here."
#
# This is an implementation of recursion in Puppet. `twitch_statsite::services`
# gets declared with a number (X) of statsite processes to be running on the server.
# This iterates X times, declaring a new statsite service each time, and dropping
# a new statsite config, incrementing the listening port by 2 each iteration.
#
# The motivation for this is due to the fact that there does not seem to be
# any other reasonable way to be able to say "I want X of these things", and
# iterate X times over a set of resources.
define twitch_statsite::services (
  $num_processes,
  $cur_iter = 0,
  ) {

  # Increment iteration variable
  $next_iter = $cur_iter + 1

  # Listening port starts at 8129 and goes up by 2 each iteration
  $port = $twitch_statsite::start_port + $cur_iter*$twitch_statsite::port_increment

  if $num_processes > 0 {
    file { "${twitch_statsite::config_path}/config_${port}.ini":
      ensure  => present,
      mode    => '0644',
      content => template('twitch_statsite/config.ini.erb'),
    }

    $pidfile = "/var/run/statsite/statsite_${port}.pid"
    twitch_monit::check { "statsite_${port}":
      pidfile        => "${pidfile}",
      cwd            => "/usr/local",
      uid            => 'statsite',
      gid            => 'statsite',
      check_interval => 5,
      command        => "${twitch_statsite::install_path}/bin/statsite.sh ${pidfile} -f /etc/statsite/config_${port}.ini 2>&1 | logger -p local3.info -t statsite_${port}",
      subscribe      => [File["/etc/statsite/config_${port}.ini"], Class["twitch_statsite::install"]],
      require        => File["/etc/statsite/config_${port}.ini"]
    }

    twitch_statsite::services { "$next_iter":
      num_processes => $num_processes-1,
      cur_iter      => $next_iter
    }
  }
}
