class twitch_unbound (
  $role                  = $twitch_unbound::params::role,
  $access_cidrs          = $twitch_unbound::params::access_cidrs,
  $allow_twitch_networks = $twitch_unbound::params::allow_twitch_networks,
  $interface             = $twitch_unbound::params::interface,
  $interface_automatic   = $twitch_unbound::params::interface_automatic,
  $val_permissive_mode   = $twitch_unbound::params::val_permissive_mode,
  $val_log_level         = $twitch_unbound::params::val_log_level,
  $num_threads           = $twitch_unbound::params::num_threads,
  $statistics_interval   = $twitch_unbound::params::statistics_interval
) inherits twitch_unbound::params {
  validate_re($role, '^(cache|resolver|forwarder|route53|video_route53|aws_forwarder)$')

  rsyslog::snippet { '43-unbound':
    ensure => 'present',
    content => "if \$programname == 'unbound' then /var/log/unbound.log\n&~\n"
  }
  file { '/var/log/archive':
    ensure  => directory,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    require => Class['::rsyslog']
  }

  logrotate::rule { 'unbound':
    ensure          => present,
    path            => '/var/log/unbound.log',
    missingok       => true,
    olddir          => '/var/log/archive',
    rotate_every    => 'day',
    rotate          => 7,
    require         => File['/var/log/archive'],
    postrotate      => 'reload rsyslog >/dev/null 2>&1 || true'
  }

  if $allow_twitch_networks {
    $_access_cidrs = concat($twitch_networks, $access_cidrs)
  } else {
    $_access_cidrs = $access_cidrs
  }

  class { '::unbound':
    interface             => $interface,
    access                => $_access_cidrs,
    conf_d                => '/etc/unbound/unbound.conf.d',
    statistics_interval   => $statistics_interval,
    extended_statistics   => 'yes',
    statistics_cumulative => true,
    interface_automatic   => $interface_automatic,
    do_ip6                => false,
    hide_identity         => true,
    hide_version          => true,
    verbosity             => 1,
    num_threads           => $num_threads,
    msg_cache_slabs       => 16,
    rrset_cache_slabs     => 16,
    infra_cache_slabs     => 16,
    key_cache_slabs       => 16,
    rrset_cache_size      => '2g',
    msg_cache_size        => '1g',
    so_rcvbuf             => '16m',
    custom_server_conf    => $custom_server_conf,
    val_permissive_mode   => $val_permissive_mode,
    val_log_level         => $val_log_level,
    require => Class['twitch_networking::dns_and_dhcp'],
  }

  twitch_monit::check{"unbound":
    pidfile        => "/var/run/unbound.pid",
    check_interval => 5,
    start_program  => "/usr/sbin/service unbound start",
    stop_program   => "/usr/sbin/service unbound stop",
    custom_checks => [
                      "if failed host 127.0.0.1 port 53 type tcp protocol dns then restart",
                      "if failed host 127.0.0.1 port 53 type udp protocol dns then restart",
                      "if 5 restarts within 5 cycles then timeout"
                      ],
    cwd => "/",
    uid => "root",
    require => Class['unbound'],
  }

  class { "twitch_unbound::${role}": }

}
