package main

import (
	"flag"
	"path/filepath"

	_ "code.justin.tv/common/golibs/bininfo"
	"code.justin.tv/kdkly/boilerplate-gen/boilerplate"

	"github.com/Sirupsen/logrus"
	"github.com/pkg/errors"
)

var (
	logLevel = flag.String("log-level", "WARN",
		"Determines output verbosity; messages less severe than the given level will not be emitted.")
	rawRepoPath = flag.String("repo-path", ".",
		"XXX: Document me.")
	rawConfigPath = flag.String("config-path", "",
		"If unspecified, the tool will look for a file named .boilerplate.yaml in the root directory of the repository.")

	// // TODO: This flag has no effect yet.
	// forceUpdateCheck = flag.Bool("force-update-check", false, "Check for an update to boilerplate-gen regardless of when the last check was.")
	updateCheckOnly = flag.Bool("update-check-only", false, "Check for an update to boilerplate-gen; take no other action.  Implies --force-update-check.")
)

// Interpret a path provided as a command-line flag; turn it into an absolute, cleaned path but do not resolve symlinks.
func interpretPath(p string) (string, error) {
	var err error
	if p == "" {
		p = "." // XXX: If this is also the default value, should we just report an error and exit here?
	}
	if p, err = filepath.Abs(p); err != nil {
		return "", errors.Wrap(err, "failed to make path absolute")
	}
	return filepath.Clean(p), nil
}
func main() {
	flag.Parse()

	l := logrus.New()
	parsedLogLevel, err := logrus.ParseLevel(*logLevel)
	if err != nil {
		l.WithError(err).Fatal("failed to parse log level")
	}
	l.Level = parsedLogLevel

	if err := boilerplate.UpdateCheck(l); err != nil {
		// TODO: Give more graceful output here.
		l.WithError(err).Fatal("failed to check for update")
	}

	if *updateCheckOnly {
		return
	}

	repoPath, err := interpretPath(*rawRepoPath)
	if err != nil {
		l.WithError(err).Fatal("failed to interpret repository path")
	}

	configPath := *rawConfigPath
	if configPath == "" {
		l.Debug("using default config path")
		configPath = filepath.Join(repoPath, boilerplate.DefaultConfigFileName)
	}
	configPath, err = interpretPath(configPath)
	if err != nil {
		l.WithError(err).Fatal("failed to interpret configuration file path")
	}

	l.WithFields(logrus.Fields{
		"repoPath":   repoPath,
		"configPath": configPath,
	}).Debug("interpreted flags")
	configFile, err := boilerplate.LoadConfigFile(configPath)
	if err != nil {
		l.WithError(err).Fatal("failed to load configuration")
	}

	l.WithFields(logrus.Fields{"configFile": configFile}).Debug("loaded configuration")

	configBundle, err := boilerplate.ConfigFromFile(configFile)
	if err != nil {
		l.WithError(err).Fatal("failed to elaborate configuration") // or: "failed to generate template parameters"?
	}

	// l.WithFields(logrus.Fields{"configBundle": configBundle}).Debug("generated template params")
	// fmt.Printf("%#v", configBundle)

	if err := configBundle.WriteBoilerplate(repoPath); err != nil {
		l.WithError(err).Fatal("failed to generate output")
	}
}
