package graffiti

import (
	"bytes"
	"encoding/json"
	"fmt"
	"strings"
	"text/template"

	"github.com/pkg/errors"
)

type Dashboard struct {
	Title string
	Rows  interface{}
}

func (d Dashboard) TemplateFile() string {
	return "./templates/dashboard.json"
}

func (d *Dashboard) MarshalJSON() ([]byte, error) {
	return Template(d, func() error {
		b, err := json.Marshal(d.Rows)
		if err != nil {
			return err
		}
		d.Rows = string(b)
		return nil
	})
}

type Row struct {
	Panels interface{}
	Title  string
}

func (r Row) TemplateFile() string {
	return "./templates/row.json"
}

func (r *Row) MarshalJSON() ([]byte, error) {
	return Template(r, func() error {
		b, err := json.Marshal(r.Panels)
		if err != nil {
			return err
		}
		r.Panels = string(b)
		return nil
	})
}

type Panel struct {
	Title      string
	Targets    interface{}
	Datasource string
}

func (p Panel) TemplateFile() string {
	return "./templates/panel.json"
}

type Templater interface {
	TemplateFile() string
}

func Template(temp Templater, f func() error) ([]byte, error) {
	t, err := template.ParseFiles(temp.TemplateFile())
	if err != nil {
		return nil, errors.Wrapf(err, "failed to parse template: %s", temp.TemplateFile())
	}

	if f != nil {
		if err := f(); err != nil {
			return nil, errors.Wrapf(err, "failed to run template func: %s", temp.TemplateFile())
		}
	}

	var buf bytes.Buffer
	err = t.Execute(&buf, temp)
	return buf.Bytes(), errors.Wrapf(err, "failed to execute template: %s", temp.TemplateFile())
}

func (p *Panel) MarshalJSON() ([]byte, error) {
	return Template(p, func() error {
		b, err := json.Marshal(p.Targets)
		if err != nil {
			return err
		}
		p.Targets = string(b)
		return nil
	})
}

type CloudwatchTarget struct {
	Alias      string            `json:"alias"`
	Dimensions map[string]string `json:"dimensions"`
	MetricName string            `json:"metricName"`
	Namespace  string            `json:"namespace"`
	Statistics []string          `json:"statistics"`
	Region     string            `json:"region"`
	Period     string            `json:"period"`
	RefID      string            `json:"refId"`
	Target     string            `json:"target"`
}

type Marshaller interface {
	MarshalJSON() ([]byte, error)
}

type Marshallers []Marshaller

func (m Marshallers) MarshalJSON() ([]byte, error) {
	var pieces []string
	for _, marshaller := range m {
		b, err := marshaller.MarshalJSON()
		if err != nil {
			return nil, err
		}

		pieces = append(pieces, string(b))
	}

	array := fmt.Sprintf("[%s]", strings.Join(pieces, ","))
	return []byte(array), nil
}
