package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"

	grafana "code.justin.tv/kevipike/graffiti"
	usersservice "code.justin.tv/kevipike/graffiti/services/users-service"
)

type ServiceConfig struct {
	Org, Service, Environment, Region string
	CloudwatchDatasource              string
	BeanstalkEnvironmentName          string
	// servers.DiamondServers
	DiamondServers string
	// stats.counters.service.env.GolangHosts
	GolangHosts string
}

func main() {
	for _, config := range []ServiceConfig{
		{
			Org: "web", Service: "users-service", Environment: "staging", Region: "us-west-2",
			CloudwatchDatasource:     "CloudWatch (twitch-web-dev)",
			BeanstalkEnvironmentName: "staging-web-users-service-server",
			DiamondServers:           "users-service-staging-ip-*",
			GolangHosts:              "ip-*",
		}, {
			Org: "web", Service: "users-service", Environment: "prod", Region: "us-west-2",
			CloudwatchDatasource:     "CloudWatch (twitch-web-aws)",
			BeanstalkEnvironmentName: "prod-web-users-service-server",
			DiamondServers:           "users-service-prod-ip-*",
			GolangHosts:              "ip-*",
		},
	} {

		var rows grafana.Marshallers

		rows = append(rows, &grafana.ServiceRequestsRow{
			Service: config.Service,
		})
		rows = append(rows, &grafana.ServiceChatRedisRow{
			Service: config.Service,
			Title:   "Redis Primary",
		})
		rows = append(rows, &grafana.ServiceIdentityRediserRow{
			Service:   config.Service,
			Namespace: "redis_elasticache",
			Title:     "Redis Backup",
		})
		rows = append(rows, &usersservice.DatabaseRow{})
		rows = append(rows, &grafana.ServiceBeanstalkRow{
			Datasource:      config.CloudwatchDatasource,
			Region:          config.Region,
			EnvironmentName: config.BeanstalkEnvironmentName,
		})
		rows = append(rows, &grafana.ServiceGolangRow{
			Service: config.Service,
		})
		rows = append(rows, &grafana.ServiceInstanceRow{
			Servers: config.DiamondServers,
		})
		rows = append(rows, &grafana.ServiceHystrixRow{
			Service: config.Service,
		})

		dashboard := &grafana.ServiceDashboard{
			Org:         config.Org,
			Service:     config.Service,
			Environment: config.Environment,
			Hosts:       config.GolangHosts,
			Rows:        rows,
		}

		b, err := json.Marshal(dashboard)
		if err != nil {
			log.Fatal(err)
		}

		filename := fmt.Sprintf("%s-%s-%s.json", config.Org, config.Service, config.Environment)
		err = ioutil.WriteFile(filename, b, 0644)
		if err != nil {
			log.Fatal(err)
		}
	}

}
