package main

import (
	"bufio"
	"io"
	"log"
	"os"
	"sort"

	"strings"

	"fmt"

	"github.com/pkg/errors"
)

var (
	modulePrefix = "users_service::"
	defaults     = map[string]string{
		"db-host-slave":                    "localhost",
		"db-port-slave":                    "5432",
		"db-max-open-conns-slave":          "100",
		"db-max-idle-conns-slave":          "50",
		"db-max-queue-size-slave":          "10",
		"db-conn-acquire-timeout-ms-slave": "1000",
		"db-request-timeout-ms-slave":      "5000",
		"db-max-conn-age-ms-slave":         "60000",

		"db-host-master":                    "localhost",
		"db-port-master":                    "5432",
		"db-max-open-conns-master":          "100",
		"db-max-idle-conns-master":          "50",
		"db-max-queue-size-master":          "10",
		"db-conn-acquire-timeout-ms-master": "1000",
		"db-request-timeout-ms-master":      "5000",
		"db-max-conn-age-ms-master":         "60000",

		"db-name":          "users_service_justintv_dev",
		"db-user":          "users_service_01",
		"db-password":      "",
		"db-password-file": ".dbpass_dev",

		"reservation-db-host-slave":                    "localhost",
		"reservation-db-port-slave":                    "5432",
		"reservation-db-max-open-conns-slave":          "100",
		"reservation-db-max-idle-conns-slave":          "50",
		"reservation-db-max-queue-size-slave":          "10",
		"reservation-db-conn-acquire-timeout-ms-slave": "1000",
		"reservation-db-request-timeout-ms-slave":      "5000",
		"reservation-db-max-conn-age-ms-slave":         "60000",

		"reservation-db-host-master":                    "localhost",
		"reservation-db-port-master":                    "5432",
		"reservation-db-max-open-conns-master":          "100",
		"reservation-db-max-idle-conns-master":          "50",
		"reservation-db-max-queue-size-master":          "10",
		"reservation-db-conn-acquire-timeout-ms-master": "1000",
		"reservation-db-request-timeout-ms-master":      "5000",
		"reservation-db-max-conn-age-ms-master":         "60000",

		"reservation-db-name":          "users_service_dev",
		"reservation-db-user":          "users_service_01",
		"reservation-db-password":      "",
		"reservation-db-password-file": ".dbpass_dev",

		"rails-host": "127.0.0.1:3000",

		"redis-host":                   "localhost",
		"redis-port":                   "14912",
		"redis-pass":                   "",
		"redis-connect-timeout-ms":     "500",
		"redis-read-timeout-ms":        "500",
		"redis-write-timeout-ms":       "100",
		"redis-conn":                   "1000",
		"redis-user-properties-ttl-ms": "30000",

		"memcached-host-ports":    "localhost:11211",
		"memcached-autodiscovery": "false",
		"memcached-conn":          "10000",
		"memcached-timeout":       "1000",

		"follows-host": "internal-following-service-staging-app-2022864404.us-west-2.elb.amazonaws.com",

		"user-mutations-stream-role":           "arn:aws:iam::465369119046:role/user-mutations-stream-role",
		"user-mutations-stream-name":           "user-mutations-stream",
		"user-mutations-stream-region":         "us-west-2",
		"user-mutations-stream-retry-count":    "10",
		"user-mutations-stream-retry-delay-ms": "100",

		"sns-region":                         "us-west-2",
		"user-moderation-events-topic-arn":   "arn:aws:sns:us-west-2:465369119046:user_moderation_events",
		"user-rename-events-topic-arn":       "arn:aws:sns:us-west-2:465369119046:user_rename_events",
		"user-creation-events-topic-arn":     "arn:aws:sns:us-west-2:465369119046:notification-user-created",
		"user-mutation-events-topic-arn":     "arn:aws:sns:us-west-2:465369119046:user_mutation_events",
		"user-image-upload-events-topic-arn": "arn:aws:sns:us-west-2:465369119046:user_image_upload_events",
		"channel-mutation-events-topic-arn":  "arn:aws:sns:us-west-2:465369119046:channel_mutation_events",
		"email-verified-topic-arn":           "arn:aws:sns:us-west-2:510557735000:email-validation-success",
		"pushy-dispatch-topic-arn":           "arn:aws:sns:us-west-2:603200399373:pushy_darklaunch_dispatch",
		"user-soft-delete-events-topic-arn":  "arn:aws:sns:us-west-2:465369119046:user_soft_delete_events",
		"user-undelete-events-topic-arn":     "arn:aws:sns:us-west-2:465369119046:user_undelete_events",
		"user-hard-delete-events-topic-arn":  "arn:aws:sns:us-west-2:465369119046:notification-user-destroyed",

		"partnerships-host": "partnerships.dev.us-west2.twitch.tv",
		"spade-host":        "spade.internal.justin.tv",
		"auditor-host":      "http://history.staging.us-west2.twitch.tv",
		"leviathan-host":    "https://leviathan.internal.twitch.tv:443",
		"leviathan-token":   "DUMMY_TOKEN",
		"rollbar-api-token": "",
		"discovery-host":    "http://discovery-staging.dev.us-west2.twitch.tv:9292",

		"legacy-aws-key":    "",
		"legacy-aws-secret": "",

		"xray-sampling": "0.01",
		"cache-backend": "redis",

		"twilio_account":       "",
		"twilio_auth":          "",
		"twilio_phone_numbers": "",
		"twilio_disabled":      "true",

		"geoip-path":       "/usr/local/share/GeoIP/GeoIP.dat",
		"cartman-key-path": "./configs/cartman_dev_key",

		"evs-host":            "https://email-validation-staging.internal.justin.tv",
		"upload-service-host": "http://staging-web-upload-service.staging.us-west2.twitch.tv",

		"sqs-region": "us-west-2",

		"email-validation-success-num-workers": "2",
		"email-validation-success-queue-name":  "email-verified",

		"owl-host": "http://owl.dev.us-west2.justin.tv",

		"noop-rails-client":                "false",
		"image-upload-success-num-workers": "2",
		"image-upload-success-queue-name":  "image-upload-staging",
		"dead-letter-queue-num-workers":    "1",
		"dead-letter-queue-name":           "users-service-dead-letter-queue-staging",
	}
)

func main() {
	if err := run(); err != nil {
		log.Fatal(err)
	}
}

type Variables map[string]Environments

func (v Variables) OrderedRange(f func(string, Environments)) {
	var keys []string
	for key := range v {
		keys = append(keys, key)
	}

	sort.Strings(keys)

	for _, key := range keys {
		f(key, v[key])
	}
}

type Environments map[string]interface{}

func run() error {
	// Parse variable names and defaults
	vars := make(Variables)

	if err := eachLine("params.pp", func(line string) {
		if !strings.HasPrefix(line, "  $") {
			return
		}

		pieces := strings.Split(line, "=")

		pieces[0] = strings.Trim(pieces[0], "$ ")
		pieces[1] = strings.Trim(pieces[1], " ")
		pieces[1] = strings.Replace(pieces[1], "'", "\"", -1)

		if strings.HasPrefix(pieces[1], "pick(") {
			return
		}

		name := strings.ToUpper(pieces[0])
		vars[name] = make(map[string]interface{})

		vars[name]["default"] = pieces[1]
	}); err != nil {
		return errors.Wrap(err, "failed to parse params line by line")
	}

	// Parse environment values from hiera/cluster
	if err := addEnvironment(vars, "production", modulePrefix); err != nil {
		return errors.Wrap(err, "failed to add production")
	}
	if err := addEnvironment(vars, "staging", modulePrefix); err != nil {
		return errors.Wrap(err, "failed to add staging")
	}

	if err := writeEnvironment(vars, "production"); err != nil {
		return errors.Wrap(err, "failed to write production")
	}
	if err := writeEnvironment(vars, "staging"); err != nil {
		return errors.Wrap(err, "failed to write staging")
	}

	return nil
}

func toDefaultCase(env string) string {
	env = strings.Replace(env, "_", "-", -1)
	return strings.ToLower(env)
}

func writeEnvironment(vars Variables, environment string) error {
	filename := fmt.Sprintf("%s.config", environment)
	f, err := os.OpenFile(filename, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	if err != nil {
		return errors.Wrapf(err, "failed to open file %q", filename)
	}
	defer f.Close()
	f.Truncate(0)
	f.Seek(0, 0)

	var totalBytes int

	fmt.Fprintln(f, "option_settings:")
	vars.OrderedRange(func(name string, options Environments) {
		value := options[environment]
		if value == nil {
			value = options["default"]
		}

		if defaults[toDefaultCase(name)] == value {
			// Do not write variables with default values in code to conserve space
			// as to not hit 4096 byte limit.
			return
		}

		fmt.Fprintf(f, "  - option_name: %s\n", name)
		fmt.Fprintf(f, "    value: %s\n", value)

		totalBytes += len([]byte(fmt.Sprintf("%s=%s", name, value)))
	})

	if totalBytes > 4096 {
		log.Println("WARNING:", environment, "is larger than 4096 bytes and won't work (", totalBytes, ")")
	}

	return nil
}

func eachLine(file string, readFunc func(string)) error {
	f, err := os.Open(file)
	if err != nil {
		return errors.Wrapf(err, "failed to read %q", file)
	}
	defer f.Close()

	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		readFunc(scanner.Text())
	}

	return scanner.Err()
}

func addEnvironment(vars Variables, environment, prefix string) error {
	return eachLine(environment, func(line string) {
		if !strings.HasPrefix(line, prefix) {
			return
		}

		line = line[len(prefix):]
		pieces := strings.SplitN(line, ":", 2)

		pieces[0] = strings.Trim(pieces[0], " ")
		pieces[1] = strings.Trim(pieces[1], " ")
		pieces[1] = strings.Replace(pieces[1], "'", "\"", -1)

		name := strings.ToUpper(pieces[0])
		if vars[name] == nil {
			log.Println(name, "is nil, wtf?")
			return
		}
		vars[name][environment] = pieces[1]
	})
}

func writeVars(vars Variables, w io.Writer) error {
	vars.OrderedRange(func(name string, options Environments) {
		fmt.Fprintf(w, "%s:\n", name)
		for env, value := range options {
			fmt.Fprintf(w, "  %s: %s\n", env, value)
		}
	})

	return nil
}
