package apiclient

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"time"
)

// TwitchClip is a clip from twitch, containing all available data
// We might want to trim this down to only the data we're interested in
type TwitchClip struct {
	BroadcasterChannelURL  string `json:"broadcaster_channel_url"`
	BroadcasterDisplayName string `json:"broadcaster_display_name"`
	BroadcasterID          string `json:"broadcaster_id"`
	BroadcasterLogin       string `json:"broadcaster_login"`
	BroadcasterLogo        string `json:"broadcaster_logo"`
	CuratorChannelURL      string `json:"curator_channel_url"`
	CuratorDisplayName     string `json:"curator_display_name"`
	CuratorID              string `json:"curator_id"`
	CuratorLogin           string `json:"curator_login"`
	PreviewImage           string `json:"preview_image"`
	Thumbnails             struct {
		Medium string `json:"medium"`
		Small  string `json:"small"`
		Tiny   string `json:"tiny"`
	} `json:"thumbnails"`
	Game          string    `json:"game"`
	CreatedAt     time.Time `json:"created_at"`
	Title         string    `json:"title"`
	ClipURL       string    `json:"clip_url"`
	ClipStatusURL string    `json:"clip_status_url"`
	Slug          string    `json:"slug"`
	Duration      float64   `json:"duration"`
	Views         int       `json:"views"`
	VodID         string    `json:"vod_id"`
	VodURL        string    `json:"vod_url"`
	EmbedURL      string    `json:"embed_url"`
	EmbedHTML     string    `json:"embed_html"`
	ID            string    `json:"id"`
	CuratorLogo   string    `json:"curator_logo,omitempty"`
}

type TwitchClipStatus struct {
	Status         string `json:"status"`
	QualityOptions []struct {
		Quality string `json:"quality"`
		Source  string `json:"source"`
	} `json:"quality_options"`
}

// ClipsResponse is the struct we get from the clips unreleased api
type ClipsResponse struct {
	Clips []TwitchClip `json:"clips"`
}

func GetClipData(username, key string, clip *TwitchClip) error {
	const clipsAPIBaseF = `https://clips.twitch.tv/api/v1/clips/%s/%s`
	bytes, err := doRequest("GET", fmt.Sprintf(clipsAPIBaseF, username, key))
	if err != nil {
		return err
	}

	return json.Unmarshal(bytes, clip)

}

func GetClipStatus(username, key string, status *TwitchClipStatus) error {
	const clipsAPIStatusF = `https://clips.twitch.tv/api/v1/clips/%s/%s/status`
	bytes, err := doRequest("GET", fmt.Sprintf(clipsAPIStatusF, username, key))
	if err != nil {
		return err
	}

	return json.Unmarshal(bytes, status)
}

func GetTrendingClips() ([]TwitchClip, error) {
	// TODO: download from actual twitch api
	file, err := ioutil.ReadFile("public/clips-test-api-response.json")
	if err != nil {
		// handle error
		logger.Error(err)
		return nil, err
	}

	var clips ClipsResponse
	err = json.Unmarshal(file, &clips)
	if err != nil {
		logger.Error(err)
		return nil, err
	}

	return clips.Clips, nil

}
