package main

import (
	"flag"
	"fmt"
	"strconv"
	"time"

	"github.com/pajlada/kkonfig"

	"code.justin.tv/kkona/kkaper/kkarrot/uploader"
	"code.justin.tv/kkona/kkaper/klogger"
	"code.justin.tv/kkona/kkaper/models"
)

// Specification KKarrot
type Specification struct {
	Database models.RedisSpecification
	Uploader uploader.UploaderSpecification
}

var logger = klogger.GetLogger()

type stringSliceFlags []string

var configPaths stringSliceFlags

func (s *stringSliceFlags) String() string {
	return fmt.Sprint(*s)
}

func (s *stringSliceFlags) Set(value string) error {
	*s = append(*s, value)
	return nil
}

func init() {
	klogger.Init()

	flag.Var(&configPaths, "config", "Config path(s)")
	flag.Parse()

	var s Specification
	err := kkonfig.Process("kkarrot", configPaths, &s)
	if err != nil {
		logger.Error(err)
	}

	models.InitRedis(s.Database)
	if err := uploader.InitS3(s.Uploader); err != nil {
		logger.Error(err)
	}
}

const pollingInterval = time.Duration(60) * time.Second

func startPoller() {
	defer func() {
		time.AfterFunc(pollingInterval, startPoller)
	}()

	if err := doPoll(); err != nil {
		logger.Errorf("doPoll returned an error, restarting at next interval. Error message: %s", err)
	}
}

func main() {
	c := make(chan bool)

	command := flag.Arg(0)
	switch command {
	case "load":
		p, err := models.CreatePlaylistFromFile("public/playlist.json")
		if err != nil {
			logger.Error(err)
			return
		}
		p.Save()
	case "addall":
		clipsToAdd := []string{
			"https://clips.twitch.tv/admiralbulldog/EnergeticBearOSsloth",
			"https://clips.twitch.tv/athenelive/BetterAlpacaBleedPurple",
			"https://clips.twitch.tv/athenelive/ClumsyLocustBionicBunion",
			"https://clips.twitch.tv/athenelive/DoubtfulCarabeefSoBayed",
			"https://clips.twitch.tv/day9tv/CuteSnakeCmonBruh",
			"https://clips.twitch.tv/day9tv/FairCurlewFreakinStinkin",
			"https://clips.twitch.tv/day9tv/MotionlessPonyKappaRoss",
			"https://clips.twitch.tv/elcgamingtv/StormyCarabeefKappa",
			"https://clips.twitch.tv/eleaguetv/AggressiveCattlePanicVis",
			"https://clips.twitch.tv/eleaguetv/ClearOwlGrammarKing",
			"https://clips.twitch.tv/eleaguetv/ClumsyFoxDogFace",
			"https://clips.twitch.tv/eleaguetv/GiftedHerdCoolCat",
			"https://clips.twitch.tv/eleaguetv/UnsightlyMartenMikeHogu",
			"https://clips.twitch.tv/eleaguetv/WrongAntelopeRiPepperonis",
			"https://clips.twitch.tv/fate_twisted_na/AgreeablePenguinUncleNox",
			"https://clips.twitch.tv/fate_twisted_na/FairPenguinKappaRoss",
			"https://clips.twitch.tv/fate_twisted_na/SplendidShrewGingerPower",
			"https://clips.twitch.tv/ghudda/RelievedChimpanzeeDOOMGuy",
			"https://clips.twitch.tv/iddqdow/BrightDogBuddhaBar",
			"https://clips.twitch.tv/iddqdow/PreciousSardineFuzzyOtterOO",
			"https://clips.twitch.tv/imaqtpie/PoorMagpieOSkomodo",
			"https://clips.twitch.tv/korea10/TenderGuanacoBCouch",
			"https://clips.twitch.tv/lifecoach1981/ShyNightingaleWinWaker",
			"https://clips.twitch.tv/lolpray8d/FrailSpiderDansGame",
			"https://clips.twitch.tv/noxious_hs/CrazyMosquitoBCouch",
			"https://clips.twitch.tv/shroud/ImpossibleTigerDuDudu",
			"https://clips.twitch.tv/shroud/RelievedChimpanzeeDBstyle",
			"https://clips.twitch.tv/spunj/PleasantDolphinBabyRage",
			"https://clips.twitch.tv/steel_tv/ZealousFalconHassaanChop",
			"https://clips.twitch.tv/trumpsc/ZanyStinkbugHumbleLife",
			"https://clips.twitch.tv/tsm_doublelift/OutstandingMouseKevinTurtle",
		}
		for _, clipURL := range clipsToAdd {
			err := models.AddPendingClip(clipURL)
			if err != nil {
				logger.Error("Error occured when adding pending clip:", err)
			}
		}
	case "add":
		clipURL := flag.Arg(1)
		if clipURL != "" {
			logger.Debug(clipURL)
			err := models.AddPendingClip(clipURL)
			if err != nil {
				logger.Error("Error occured when adding pending clip:", err)
			}
		} else {
			fmt.Println("Usage: ./kkarrot add CLIPURL")
		}
	case "pending":
		i, err := strconv.Atoi(flag.Arg(1))
		if err != nil {
			fmt.Println("Usage: ./kkarrot pending COUNT")
			return
		}

		pendingClips, err := models.GetPendingClips(i)
		if err != nil {
			logger.Error("Error occured when getting pending clips:", err)
			return
		}

		logger.Debug(pendingClips)
	case "nothing":
	case "wait":
		<-c
	case "poll":
		if err := doPoll(); err != nil {
			logger.Error(err)
		}
	default:
		fallthrough
	case "start":
		startPoller()
		<-c
	}
}
