package models

import (
	"encoding/json"
	"time"

	"code.justin.tv/kkona/kkaper/transcoder"

	"github.com/garyburd/redigo/redis"
)

type Clip struct {
	// Main identifier from a clip. Required for load/save to work
	Slug string

	// Data loaded from the clips api
	ID                            string
	ClipURL                       string
	Duration                      float64
	VodID                         string
	VodURL                        string
	BroadcasterChannelURL         string
	BroadcasterChannelDisplayName string
	BroadcasterID                 string

	// Data filled in when a clip has been uploaded to the bucket
	BucketPath string

	// URL to broadcaster logo?
	BroadcasterLogo string
	CreatedAt       time.Time
}

const clipsKey = "kkarrot:clips"

func (c *Clip) Load() bool {
	conn := db.Get()
	defer conn.Close()

	bytes, err := redis.Bytes(conn.Do("HGET", clipsKey, c.Slug))
	if err != nil {
		logger.Error("Error when doing HGET on", c.Slug, err)
		return false
	}

	// XXX(pajlada): Not sure if this is dangerous to do
	err = json.Unmarshal(bytes, c)

	return err == nil
}

func (c *Clip) Save() (err error) {
	conn := db.Get()
	defer conn.Close()

	bytes, err := json.Marshal(c)
	if err != nil {
		return err
	}

	_, err = conn.Do("HSET", clipsKey, c.Slug, bytes)

	return err
}

func (c *Clip) CreateClipJob() transcoder.ClipJob {
	return transcoder.ClipJob{
		ClipURL: c.ClipURL,
	}
}

// forsenE
func GetAllClips() (clips []Clip, err error) {
	conn := db.Get()
	defer conn.Close()

	byteSlices, err := redis.ByteSlices(conn.Do("HGETALL", clipsKey))
	if err != nil {
		return nil, err
	}

	// HGETALL returns two slices for each element:
	// Value, then key
	numClips := len(byteSlices) / 2

	clips = make([]Clip, numClips, numClips)

	n := 0
	for i, clipJsonBytes := range byteSlices {
		if i%2 == 0 {
			continue
		}
		// YOLO
		json.Unmarshal(clipJsonBytes, &clips[n])
		n++
	}

	return clips, err
}
