package transcoder

import (
	"bufio"
	"fmt"
	"os/exec"
)

var (
	// ActiveTranscoders is a buffered channel where the size specifies how many transcodes can be active at the same time
	ActiveTranscoders = make(chan bool, 5)
)

const debugOutput = false

// ClipJob describes a transcoding clip job
type ClipJob struct {
	ClipURL        string
	OutputFileName string
}

// StartJob tries to start a transcoder job if there's room in the channel
func StartJob(job ClipJob) error {
	// Wait for buffered channel if there's no room
	ActiveTranscoders <- true

	return fireFFmpegClip(job)
}

// fireFFmpegClip starts working on the given ClipJob.
func fireFFmpegClip(job ClipJob) (err error) {
	defer func() {
		<-ActiveTranscoders
	}()

	//thanks jorge
	ffmpegArgs := []string{
		"-y",
		"-protocol_whitelist", "file,http,https,tcp,tls",
		"-i", job.ClipURL,
		"-vcodec", "libx264",
		"-b:v", "8000k",
		"-preset", "ultrafast",
		"-pix_fmt", "yuv420p",
		"-r", "60",
		"-acodec", "aac",
		"-b:a", "192k",
		"-ar", "48000",
		"-strict", "-2",
		"-threads", "6",
		job.OutputFileName,
	}

	cmd := exec.Command("ffmpeg", ffmpegArgs...)

	stderr, _ := cmd.StderrPipe()
	in := bufio.NewScanner(stderr)

	if err = cmd.Start(); err != nil {
		return err
	}

	for in.Scan() {
		if debugOutput {
			fmt.Println(in.Text())
		}
	}

	err = cmd.Wait()

	return err
}
