var _ = require("lodash");
var settings = require("./setting.json");
var s3 = require('s3');
var fileExtension = require('file-extension');
var fs = require('fs');
var md5 = require('md5');
var jsonfile = require('jsonfile');

var cachedFiles = [];

var exports = module.exports = {};

var client = s3.createClient({
    maxAsyncS3: 20,     // this is the default
    s3RetryCount: 3,    // this is the default
    s3RetryDelay: 1000, // this is the default
    multipartUploadThreshold: 20971520, // this is the default (20 MB)
    multipartUploadSize: 15728640, // this is the default (15 MB)
    s3Options: settings.S3,
});

var CachedFile = function (path, cb, persistent) {
    var _this = this;
    this.path = path;
    this.triggerPath = md5(path);
    this.localPath =  this.triggerPath + "." + fileExtension(path);
    this.isDownloading = true;
    this.lastTouched = Date.now();
    this.persistent = (typeof persistent !== 'undefined') ? persistent:false;

    this.downloader = null;

    if (!fs.existsSync(settings.cacheFolder + this.localPath)) {
        var params = {
            localFile: settings.cacheFolder + this.localPath,

            s3Params: {
                Bucket: settings.S3Bucket,
                Key: path
            }
        };
        this.downloader = client.downloadFile(params);
        this.downloader.on('error', function(err) {
            console.error("unable to download:", err.stack);
            _.remove(cachedFiles, function (o) {
                return o.path = _this.path;
            });
            _this.isDownloading = false;

            console.log("ERRORRRRRR");
        });
        this.downloader.on('progress', function() {
            console.log("progress", downloader.progressAmount, downloader.progressTotal);
        });
        this.downloader.on('end', function() {
            _this.isDownloading = false;
            cb();
        });
    } else {
        cb();
    }

    cachedFiles[this.path] = this;

    this.delete = function () {
        fs.unlink(settings.cacheFolder + this.localPath, function (err) {
            if (err) throw err;
            console.log('Deleted ' + settings.cacheFolder + _this.localPath + " from cache because it went stale.");
        });
    }
}

var peekFile = exports.peekFile = function (path, cb) {
    var find = cachedFiles[path];

    if (find) {
        find.lastTouched = Date.now();
        cb();
    } else {
        find = new CachedFile(path, cb);
    }

    return find.triggerPath;

}

exports.filterAssets = function (assets) {
    return _.filter(assets, function (asset) {
        return peekFile(asset.path);
    });
}

exports.waitForAssets = function (assets, cb) {
    var counter = assets.length;

    if (counter == 0) {
        setTimeout(cb, 0);
    } else {
        _.forEach(assets, function (o) {
            peekFile(o.path, function () {
                if (--counter == 0) {
                    cb();
                }
            });
        });
    }
}

exports.waitForStacks = function (stacks, cb) {
    var counter = stacks.length;

    if (counter == 0) {
        setTimeout(cb, 0);
    } else {
        _.forEach(stacks, function (s) {
            exports.waitForAssets(s.elements, function () {
                counter--;
                if (counter == 0) {
                    cb();
                }
            });
        });
    }
}

setInterval(function () {
    var removed = _.remove(cachedFiles, function (file) {
        if (file.isDownloading || file.persistent) {
            return false;
        } else {
            return (file.lastTouched < (Date.now() - settings.cacheTime));
        }
    });

    _.forEach(removed, function (e) {
        e.delete();
    });
}, 60 * 1000);
