package api

import (
	"fmt"
	"net/http"

	"github.com/gorilla/mux"
)

type Api struct {
	conf       Config
	fileServer http.Handler
	stats      Statter
}

func NewApi(conf Config, stats Statter) *Api {
	return &Api{
		conf:       conf,
		fileServer: http.FileServer(http.Dir(conf.STATIC_ASSETS_DIR)),
		stats:      stats,
	}
}

func (a *Api) SetupRoutes(r *mux.Router) {
	// RBAC reverse proxy
	r.Handle("/twirp/code.justin.tv.devrel.rbac.RBAC/{method}",
		NewReverseProxy("RBAC", a.conf.RBAC_HOST, a.stats))

	// RBACAdmin reverse proxy
	r.Handle("/twirp/code.justin.tv.devrel.rbac.RBACAdmin/{method}",
		NewReverseProxy("RBACAdmin", a.conf.RBAC_HOST, a.stats))

	// RBACActionHistory reverse proxy
	r.Handle("/twirp/code.justin.tv.devrel.rbac.RBACActionHistory/{method}",
		NewReverseProxy("RBACActionHistory", a.conf.RBAC_HOST, a.stats))

	// ExtensionReviews reverse proxy
	r.Handle("/twirp/code.justin.tv.devrel.rbac.ExtensionReviews/{method}",
		NewReverseProxy("ExtensionReviews", a.conf.RBAC_HOST, a.stats))

	// TwitchE2IngestHTTP reverse proxy lambda
	r.Handle("/twirp/twitch.fulton.example.twitche2ingesthttp.TwitchE2IngestHTTP/{method}",
		NewReverseProxyTwirpLambda("TwitchE2IngestHTTP", a.conf.E2_INGEST_HTTP_LAMBDA, a.stats))

	// catch all for invalid twirp routes
	r.PathPrefix("/twirp").HandlerFunc(a.TwirpBadRoute)

	// Load balancer healthcheck
	r.HandleFunc("/health", a.Health)

	// Static assets (index.html)
	r.PathPrefix("/").HandlerFunc(a.ServeStaticAssets)
}

// Health handles the ALB healthcheck responding with a 200 status
func (a *Api) Health(w http.ResponseWriter, r *http.Request) {
	SkipRequestLog(r) // keep access logs decluttered
	fmt.Fprint(w, "OK")
}

func (a *Api) TwirpBadRoute(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("ContentType", "application/json")
	w.WriteHeader(404)
	fmt.Fprint(w, `{"code": "bad_route", "msg": "Invalid Twirp service or route"}`)
}
