package api

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"

	twirplambdatransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
)

// NewReverseProxyTwirpLambda builds a reverse proxy that converts incoming requests
// into Fulton-Twirp-Lambda invocations. The incoming HTTP requests should have been made
// by a Twirp client from the destination service. The returned *ReverseProxyTwirpLambda implements http.Handler.
// The serviceName is used to identify the downstream service on logs and errors.
// The lambdaARN identies the Lambda (e.g. "arn:aws:lambda:us-west-2:123456789012:function:haberdasher").
func NewReverseProxyTwirpLambda(serviceName, lambdaARN string, stats Statter) *ReverseProxyTwirpLambda {
	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Region:     aws.String("us-west-2"),
			HTTPClient: &http.Client{Timeout: 10 * time.Second},
		},
	}))
	lambdaCli := lambda.New(sess)
	transportCli := twirplambdatransport.NewClient(lambdaCli, lambdaARN)

	return &ReverseProxyTwirpLambda{
		serviceName:  serviceName,
		transportCli: transportCli,
		stats:        stats,
	}
}

type ReverseProxyTwirpLambda struct {
	serviceName  string
	transportCli *twirplambdatransport.Client
	stats        Statter
}

// ServeHTTP is the method that implements http.Handler, so it can be mounted in a router mux.
// It sends the same request through the lambda-transport client, which ignores the request host,
// because it is actually using AWS Go SDK to make a lambda invocation based on the lambdaName.
func (rp *ReverseProxyTwirpLambda) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	AddRequestLogKeyStr(r, "serviceName", rp.serviceName)

	var err error
	defer func() {
		if err != nil {
			AddRequestLogKeyErr(r, err)
			rp.stats.Inc(fmt.Sprintf("ProxyClient.%s.%s", rp.serviceName, statusGroupXxx(500)), 1) // e.g. "ProxyClient.TwitchE2IngestHTTP.5xx"
			w.WriteHeader(500)
			_, _ = w.Write([]byte(`{"code": "internal", "msg": "lambda: proxy error"}`))
		}
	}()

	resp, err := rp.transportCli.Do(r)
	if err != nil {
		return
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	rp.stats.Inc(fmt.Sprintf("ProxyClient.%s.%s", rp.serviceName, statusGroupXxx(resp.StatusCode)), 1) // e.g. "ProxyClient.TwitchE2IngestHTTP.2xx"
	w.WriteHeader(resp.StatusCode)
	_, _ = w.Write(body)
}
