package api

import (
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
)

type Statter interface {
	// Inc reports "Count" to CloudWatch
	Inc(metric string, n int)

	// Duration reports "Seconds" to CloudWatch
	Duration(metric string, d time.Duration)
}

func NewTelemetryStatter(tPid identifier.ProcessIdentifier) *TelemetryStatter {
	sender := cw.NewUnbuffered(&tPid, nil)
	sampleBuilder := telemetry.SampleBuilder{ProcessIdentifier: tPid}
	sampleObserver := telemetry.NewBufferedAggregator(20*time.Second, 100000, time.Minute, sender, nil)

	return &TelemetryStatter{
		r: &telemetry.SampleReporter{
			SampleBuilder:  sampleBuilder,
			SampleObserver: sampleObserver,
		},
	}
}

type TelemetryStatter struct {
	r *telemetry.SampleReporter
}

func (s *TelemetryStatter) Inc(metric string, n int) {
	s.r.Report(metric, float64(n), telemetry.UnitCount)
}
func (s *TelemetryStatter) Duration(metric string, d time.Duration) {
	s.r.ReportDurationSample(metric, d)
}

// NullStatter can be used for tests and local development
type NullStatter struct{}

func (s *NullStatter) Inc(metric string, n int)                {}
func (s *NullStatter) Duration(metric string, d time.Duration) {}
func (s *NullStatter) Gauge(metric string, n int)              {}
