#!/usr/bin/env bash
set -e

# Requirements:
#  * Build docker image first (make build)
#  * aws-cli v2
#  * jq (install with brew)
#  * mwinit (and access to Amazon VPN): https://wiki.twitch.com/display/AD/Using+Isengard+from+the+command+line+or+programatically
#  * ada credentials: install with: toolbox install ada

show_usage_and_exit() {
  echo "Usage: $0 <env>"
  echo "  <env>: dev|prod"
  echo "Example: $0 dev"
  exit 1
}

ENV_NAME=$1
case $ENV_NAME in
  "dev")
    export AWS_PROFILE="vienna-dev"
    export AWS_ACCOUNT="637201527533"
    export AWS_DEFAULT_REGION="us-west-2"
    ECS_CLUSTER="Dev-EcsViennaStack-ClusterEB0386A7-7Ock4vGEEmqM"
    ECS_SERVICE="Vienna"
    ;;
  "prod")
    export AWS_PROFILE="vienna-prod"
    export AWS_ACCOUNT="789746331623"
    export AWS_DEFAULT_REGION="us-west-2"
    ECS_CLUSTER="Prod-EcsViennaStack-ClusterEB0386A7-oyMj41lsSNLQ"
    ECS_SERVICE="Vienna"
    ;;
  *)
    echo "invalid <env>: $1"
    show_usage_and_exit
    ;;
esac

# TODO: use GIT_TAG instead of :latest
# GIT_TAG=$2
# if [[ -z $GIT_TAG ]]; then
#   GIT_TAG=`git log -1 --pretty=%H`
# fi

DOCKER_IMAGE_NAME="vienna" # name used to build the container (docker build -t <name>)
DOCKER_IMAGE_TAG="latest"
ECR_REPO="vienna" # the name of the ECR repository to push

echo "Auth: Make sure to run mwinit from WPA2 or AmazonVPN, and run ada (copy-pasta this):"
echo "      ada credentials update --once --provider=isengard --role=admin --profile=$AWS_PROFILE --account=$AWS_ACCOUNT"
echo ""

echo ">> Get Docker image id for tag: ${DOCKER_IMAGE_NAME}"
DOCKER_IMAGE_ID=`docker images ${DOCKER_IMAGE_NAME}:$DOCKER_IMAGE_TAG --format "{{.ID}}"`
if [ -z "$DOCKER_IMAGE_ID" ]; then
    echo "docker images ${DOCKER_IMAGE_NAME}:$DOCKER_IMAGE_TAG did not return any result. Please, make sure to do first: docker build -t ${DOCKER_IMAGE_NAME} ."
    exit 1
fi
echo "DOCKER_IMAGE_ID=$DOCKER_IMAGE_ID"

# Push image to ECR: https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-push-ecr-image.html
echo ">> Docker login to authenticate your Docker CLI to the ECR registry"
aws ecr get-login-password ${REGION} | docker login --username AWS --password-stdin ${AWS_ACCOUNT}.dkr.ecr.us-west-2.amazonaws.com

echo ">> Tag image with the Amazon ECR registry"
ECR_REGISTRY_TAG=$AWS_ACCOUNT.dkr.ecr.us-west-2.amazonaws.com/$ECR_REPO:$DOCKER_IMAGE_TAG
docker tag $DOCKER_IMAGE_ID $ECR_REGISTRY_TAG

echo ""
echo "docker push $ECR_REGISTRY_TAG"
docker push $ECR_REGISTRY_TAG

echo ">> Blue/Green deploy to ECS and wait until all tasks restart"
echo "./scripts/ecs-deploy.sh -p $AWS_PROFILE -r $AWS_DEFAULT_REGION --cluster \"$ECS_CLUSTER\" --service-name \"$ECS_SERVICE\" --image \"$ECR_REGISTRY_TAG\" --timeout 60000"
./scripts/ecs-deploy.sh -p $AWS_PROFILE -r $AWS_DEFAULT_REGION --cluster "$ECS_CLUSTER" --service-name "$ECS_SERVICE" --image "$ECR_REGISTRY_TAG" --timeout 60000
