import { get as getCookie } from 'js-cookie';

/**
 * Thin wrapper around fetch; simply adds an 'Authorization' header containing
 * the OAuth token of the logged-in user from redux, overwriting any existing auth header if present.
 * Direct pass-through to fetch if there is no logged-in user.
 */
export function fetchWithAuth(input: RequestInfo, init?: RequestInit): Promise<Response> {
  const sessionUser = getCookie('vienna_auth_token');

  // No authed user means no auth header to add.
  if (!sessionUser) {
    return fetch(input, init);
  }

  const headers = init ? init.headers : {};
  let newHeaders;

  if (Array.isArray(headers)) {
    newHeaders = headers.slice();

    const authHeaderIndex = newHeaders.findIndex((header) => header[0] === 'Authorization');
    if (authHeaderIndex >= 0) {
      newHeaders.splice(authHeaderIndex, 1);
    }

    newHeaders.push(['Authorization', authHeaderValue(sessionUser)]);
    // TODO: Why don't these work here??
    newHeaders.push(['Accept', 'application/json']);
    newHeaders.push(['Content-Type', 'application/json']);
  } else if (headers instanceof Headers) {
    newHeaders = new Headers(headers);
    newHeaders.set('Authorization', authHeaderValue(sessionUser));
    newHeaders.set('Accept', 'application/json');
    newHeaders.set('Content-Type', 'application/json');
  } else {
    newHeaders = {
      ...headers,
      Authorization: authHeaderValue(sessionUser),
      Accept: 'application/json',
      'Content-Type': 'application/json',
    };
  }

  return fetch(input, { ...init, headers: newHeaders });
}

export function authHeaderValue(token: string) {
  return `OAuth ${token}`;
}
