import React from 'react';
import { useFormik } from 'formik';
import { Layout, Display } from 'twitch-core-ui';
import {
  ViennaFormConfig,
  ViennaFormValues,
  ViennaFormValueType,
  ViennaFormField,
} from '~core/components/vienna-form/models/form-model';
import {
  ViennaStringFormInput,
  ViennaSelectForminput,
  ViennaBooleanInputField,
} from '~core/components/vienna-form/components';

export interface PublicProps<T extends ViennaFormValues> {
  initialValues: T;
  onSubmit: (values: T) => void;
  form: ViennaFormConfig;
}

type Props<T extends ViennaFormValues> = PublicProps<T>;

export const ViennaForm = <T extends ViennaFormValues>(props: React.PropsWithChildren<Props<T>>) => {
  const formik = useFormik<T>({
    initialValues: props.initialValues,
    onSubmit: props.onSubmit,
  });

  const renderField = (field: ViennaFormField, fieldIndex: number) => {
    switch (field.type) {
      case ViennaFormValueType.String:
        return (
          <ViennaStringFormInput
            value={formik.values[field.name] as string}
            formField={field}
            onChange={formik.handleChange}
            onBlur={formik.handleBlur}
            key={`field-${fieldIndex}`}
          />
        );
      case ViennaFormValueType.Select:
        return (
          <ViennaSelectForminput
            value={formik.values[field.name] as string}
            formField={field}
            onChange={formik.handleChange}
            onBlur={formik.handleChange}
            key={`field=${fieldIndex}`}
          />
        );
      case ViennaFormValueType.Bool:
        return (
          <ViennaBooleanInputField
            value={formik.values[field.name] as boolean}
            formField={field}
            onChange={formik.handleChange}
            onBlur={formik.handleBlur}
            key={`field-${fieldIndex}`}
          />
        );
      default:
        throw new Error('Invalid Field');
    }
  };

  return (
    <Layout>
      <form onSubmit={formik.handleSubmit}>
        {props.form.map((fields, fieldsIndex) => {
          return (
            <Layout display={Display.Flex} key={`fields-${fieldsIndex}`} margin={{ bottom: 2 }}>
              {fields.map((field, fieldIndex) => {
                return renderField(field, fieldIndex);
              })}
            </Layout>
          );
        })}
      </form>
    </Layout>
  );
};
