import React from 'react';
import {
  ModalHeader,
  FormGroup,
  CoreImage,
  ButtonIcon,
  SVGAsset,
  ModalSize,
  Input,
  InputType,
  Layout,
  StyledLayout,
  BorderRadius,
  CoreText,
  Display,
  FlexWrap,
  AlignItems,
  Radio,
  ModalFooter,
} from 'twitch-core-ui';
import { Modal, ModalId } from 'tachyon-modal';
import { Company } from '~core/clients/rbac/code.justin.tv.devrel';
import { CompanyAutocomplete } from '~features/company-autocomplete/company-autocomplete';
import { GameAutocomplete } from '~features/game-autocomplete';
import { Game } from '~features/game-autocomplete/models/game';
import { AutocompleteChildProps } from '~features/autocomplete/autocomplete';
import {
  SearchGames_searchFor_games_items,
  getGameDetails_game,
} from '~core/graphql/twitch-gql-schema';
import { Product, AllowlistEntry } from '~core/clients/twitch-e2-ingest-http/twitch.fulton.example';
import { useFormik } from 'formik';
import './styles.scss';
import * as yup from 'yup';
import { Products } from '~features/e2/product';
import { getGameDetails, getGameDetailsVariables } from '~core/graphql/twitch-gql-schema';
import getGameDetailsQuery from '~pages/games-review-detail/queries/get-game-details.gql';
import { useQuery } from '@apollo/react-hooks';

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  onSubmit: (values: E2ModalSchema) => void;
  active: boolean;
  entry?: AllowlistEntry; // if provided, we are editing the allowlist entry
  organization?: Company;
}

type Props = PublicProps;

export interface E2ModalSchema {
  organizationId: string;
  gameId: string;
  clientId: string;
  product?: Product;
}

export const AllowlistE2Modal: React.FC<Props> = (props) => {
  const { data } = useQuery<getGameDetails, getGameDetailsVariables>(getGameDetailsQuery, {
    variables: { id: props.entry?.gameId || '' },
    skip: !props.active || !props.entry,
  });

  const productSchema = yup.mixed<Product>().label('Product');
  const formValidation = yup.object<E2ModalSchema>({
    organizationId: yup.string().label('Organization').required(),
    gameId: yup.string().label('Game').required(),
    clientId: yup.string().label('Client ID').required(),
    product: props.entry
      ? productSchema.notRequired()
      : productSchema
          .transform((product) => (product === Product.UNKNOWN ? undefined : product))
          .required(),
  });

  const formik = useFormik<E2ModalSchema>({
    onSubmit: onSubmit,
    initialValues: {
      organizationId: props.organization?.id || '',
      gameId: props.entry?.gameId || '',
      clientId: props.entry?.clientId || '',
      product: props.entry?.product || Product.UNKNOWN,
    },
    validationSchema: formValidation,
    enableReinitialize: true,
  });

  function onSubmit(values: E2ModalSchema) {
    props.onSubmit(values);
    formik.resetForm();
    props.closeModal();
  }

  function closeModal() {
    formik.resetForm();
    props.closeModal();
  }

  function onCompanySelect(company?: Company) {
    formik.setFieldValue('organizationId', company?.id);
  }

  function onGameSelect(game: Game) {
    formik.setFieldValue('gameId', game?.id);
  }

  if (!props.active) {
    return null;
  }

  return (
    <Modal id={props.id}>
      <ModalHeader
        size={ModalSize.Small}
        title={props.entry ? 'Edit E2 Client' : 'Allowlist E2 Client'}
        padding={0.5}
        closeButton={{ onClick: closeModal, 'aria-label': 'Close' }}
      />
      <StyledLayout padding={{ x: 2, top: 2, bottom: 1 }} className="e2-allowlist__modal">
        {(props.entry || !props.organization) && (
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Organization"
              error={!!formik.errors.organizationId}
              errorMessage={formik.errors.organizationId}
            >
              <CompanyAutocomplete
                onCompanySelect={onCompanySelect}
                selectedCompany={props.organization}
              />
            </FormGroup>
          </Layout>
        )}
        <Layout margin={{ bottom: 2 }}>
          <FormGroup
            label="Client ID"
            hint="Generated by a developer in the organization"
            error={!!formik.errors.clientId}
            errorMessage={formik.errors.clientId}
          >
            <Input
              type={InputType.Text}
              name="clientId"
              value={formik.values.clientId}
              onChange={formik.handleChange}
              disabled={!!props.entry}
            ></Input>
          </FormGroup>
        </Layout>
        <Layout margin={{ bottom: 2 }}>
          <FormGroup
            label="Game"
            error={!!formik.errors.gameId}
            errorMessage={formik.errors.gameId}
          >
            {props.entry ? (
              <>
                <Input
                  type={InputType.Text}
                  name="gameId"
                  disabled={true}
                  value={props.entry.gameName}
                />
                {gameBoxArt(data?.game || undefined)}
              </>
            ) : (
              <GameAutocomplete
                onGameSelect={onGameSelect}
                child={(props: AutocompleteChildProps<SearchGames_searchFor_games_items>) => {
                  return gameBoxArt(
                    props.selectedItem,
                    <ButtonIcon icon={SVGAsset.Close} onClick={props.reset} />,
                  );
                }}
              />
            )}
          </FormGroup>
        </Layout>
        <Layout>
          <FormGroup
            label="Product"
            error={!!formik.errors.product}
            errorMessage={formik.errors.product}
          >
            {Array.from(Products).map(([product, label]) => {
              if (product === Product.UNKNOWN) {
                return null;
              }
              return (
                <Layout margin={{ y: 0.5 }} key={label}>
                  <Radio
                    name="product"
                    label={label}
                    value={product}
                    onChange={formik.handleChange}
                    defaultChecked={props.entry ? props.entry.product === product : false}
                  />
                </Layout>
              );
            })}
          </FormGroup>
        </Layout>
      </StyledLayout>
      <ModalFooter
        primaryButtonProps={{
          children: 'Submit',
          fullWidth: true,
          onClick: formik.submitForm,
          disabled: !formik.isValid,
        }}
        secondaryButtonProps={{
          children: 'Cancel',
          fullWidth: true,
          onClick: closeModal,
        }}
        size={ModalSize.Small}
        padding={0.5}
      />
    </Modal>
  );
};

const gameBoxArt = (
  game?: SearchGames_searchFor_games_items | getGameDetails_game,
  children?: React.ReactNode,
) => {
  if (game && game.boxArtURL) {
    return (
      <StyledLayout
        display={Display.Flex}
        flexWrap={FlexWrap.NoWrap}
        alignItems={AlignItems.Center}
        margin={{ y: 1 }}
        padding={1}
        border
        borderRadius={BorderRadius.Large}
      >
        <Layout className="e2-allowlist__box-art__image">
          <CoreImage src={game.boxArtURL} alt={game.boxArtURL} />
        </Layout>
        <Layout padding={1} flexGrow={1}>
          <CoreText>{game.displayName}</CoreText>
        </Layout>
        {children}
      </StyledLayout>
    );
  } else {
    return null;
  }
};
