import React, { useState } from 'react';
import { ModalId, Modal, ModalSize } from 'tachyon-modal';
import { config } from '~core/vienna';
import {
  Button,
  ButtonState,
  ButtonType,
  CoreText,
  FontSize,
  FormGroup,
  InjectLayout,
  Layout,
  ModalFooterProps,
  Position,
  SVGAsset,
  TextAlign,
  ModalHeader,
  ModalFooter,
  ModalBody,
} from 'twitch-core-ui';
import { RBACAdminClient } from '~core/vienna';
import {
  CreateShadowAccountRecord,
  CreateShadowAccountRequest,
  CreateShadowAccountUserInput,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';
import { InfoField } from 'src/layout/info-field';

export const testSelector = {
  downloadTemplateBtn: 'download-csv-btn',
  uploadCSVBtn: 'upload-csv-btn',
};

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  companyId: string;
}

export type Props = PublicProps;

export interface CSVRow {
  firstname: string;
  lastname: string;
  title: string;
  email: string;
}

const header: CSVRow = {
  firstname: 'firstname (optional)',
  lastname: 'lastname (optional)',
  title: 'title (optional)',
  email: 'email (required)',
};

const responseHeader = new CreateShadowAccountRecord({
  email: 'email',
  twitchId: 'twitch ID',
  username: 'username',
  errorMessage: 'error message',
});

export interface State {
  errorMessage: string;
  file: File | null;
  csvRows: CSVRow[];
  pendingCreation: boolean;
}

export const CreateShadowAccountModal: React.FC<Props> = (props: Props) => {
  const [errorMessage, setErrorMessage] = useState<string>('');
  const [file, setFile] = useState<File | null>(null);
  const [csvRows, setCsvRows] = useState<CSVRow[]>([]);
  const [pendingCreation, setPendingCreation] = useState(false);
  const buttonState = pendingCreation ? ButtonState.Loading : ButtonState.Default;

  const shouldDisableGenerateButton = csvRows.length === 0 || !!errorMessage;

  const handleFileUpload = async (e: React.FormEvent<HTMLInputElement>) => {
    const files: FileList | null = (e.target as HTMLInputElement).files;
    setErrorMessage('');
    if (files && files[0]) {
      const uploadedFile: File = files[0];
      setFile(uploadedFile);

      const fileReader = new FileReader();
      fileReader.onloadend = () => {
        const result = fileReader.result;
        if (result) {
          const content: string = result.toString();
          const parsedCSVRows = convertFileContentToCSVRows(content);
          setCsvRows(parsedCSVRows);
        }
      };
      fileReader.readAsText(uploadedFile as Blob);
    }
  };

  const createShadowAccounts = async () => {
    const clientID = config.clientId;
    if (csvRows.length === 0) {
      return;
    }
    setPendingCreation(true);
    setErrorMessage('');
    try {
      const { records, error } = await RBACAdminClient.createShadowAccount(
        new CreateShadowAccountRequest({
          companyId: props.companyId,
          clientId: clientID,
          userInput: csvRows.map((row) => {
            return new CreateShadowAccountUserInput({
              firstName: row.firstname,
              lastName: row.lastname,
              email: row.email,
              title: row.title,
            });
          }),
        }),
      );
      setPendingCreation(false);
      downloadRecords(records);
      if (error || !records) {
        const errMsg = !error
          ? 'Some errors happened, please check the csv file for details'
          : error;
        setErrorMessage(errMsg);
        toast(`Failed to create shadow accounts: ${errMsg}`);
      }
    } catch (err) {
      setPendingCreation(false);
      toast(`Failed to create shadow accounts: ${(err as Error).message}`);
    }
  };
  const renderModalBody = () => {
    return (
      <Layout padding={{ x: 1, y: 1 }} textAlign={TextAlign.Left}>
        <FormGroup
          label=""
          error={!!errorMessage}
          errorMessage={errorMessage}
          hint={file ? file.name : ''}
        >
          <Layout>
            <InfoField
              title="Shadow Accounts"
              value="Shadow accounts allow people in an organization to test their products on Twitch
              safely. Emails provided for the creation of shadow accounts need to be tied to people
              within the requesting organization."
            />
          </Layout>
          <Layout margin={{ top: 2 }}>
            <CoreText bold fontSize={FontSize.Size5}>
              CSV Template
            </CoreText>
            <Button
              icon={SVGAsset.Download}
              type={ButtonType.Secondary}
              onClick={handleDownloadTemplate}
              download="template.csv"
              data-test-selector={testSelector.downloadTemplateBtn}
            >
              Download CSV Template
            </Button>
          </Layout>
          <Layout margin={{ top: 2 }} position={Position.Relative}>
            <CoreText bold fontSize={FontSize.Size5}>
              Upload File
            </CoreText>
            <Button icon={SVGAsset.Upload} type={ButtonType.Primary}>
              Upload CSV
            </Button>
            <InjectLayout position={Position.Absolute} fullWidth fullHeight attachLeft attachTop>
              <input
                data-test-selector={testSelector.uploadCSVBtn}
                style={{ opacity: 0 }}
                disabled={false}
                accept=".csv"
                type="file"
                onChange={handleFileUpload}
              />
            </InjectLayout>
          </Layout>
        </FormGroup>
      </Layout>
    );
  };

  const footerProps: ModalFooterProps = {
    primaryButtonProps: {
      children: 'Generate',
      disabled: shouldDisableGenerateButton,
      onClick: createShadowAccounts,
      state: buttonState,
    },
    secondaryButtonProps: {
      children: 'Cancel',
      onClick: props.closeModal,
    },
  };
  return (
    <Modal id={props.id} size={ModalSize.Small}>
      <ModalHeader
        closeButton={{ onClick: props.closeModal, 'aria-label': 'Close' }}
        title="Add Shadow Accounts"
      />
      <ModalBody>{renderModalBody()}</ModalBody>
      <ModalFooter {...footerProps} />
    </Modal>
  );
};

function handleDownloadTemplate() {
  const exampleRow: CSVRow = {
    firstname: 'example_firstname',
    lastname: 'example_lastname',
    title: '',
    email: 'example@test.com',
  };
  const dataRows = [header, exampleRow];
  const csvString = dataRows
    .map((row) => {
      return convertToCSVString(row);
    })
    .join('');
  const url = generateCVSDownloadLink(csvString);
  window.location.assign(url);
}

function convertToCSVString(obj: CSVRow): string {
  return `${obj.firstname},${obj.lastname},${obj.title},${obj.email}\n`;
}

function convertRecordToCSVString(obj: CreateShadowAccountRecord): string {
  return `${obj.email},${obj.twitchId},${obj.username},${obj.errorMessage || ''}\n`;
}

function generateCVSDownloadLink(csvString: string): string {
  const blob = new Blob([csvString], { type: 'text/csv;charset=utf-8;' });
  return URL.createObjectURL(blob);
}

function convertStringToCSVRow(text: string): CSVRow {
  const values: string[] = text.split(',');
  return {
    firstname: values[0].trim(),
    lastname: values[1].trim(),
    title: values[2].trim(),
    email: values[3].trim(),
  };
}

function convertFileContentToCSVRows(content: string | undefined): CSVRow[] {
  if (!content) {
    return [];
  }
  const rows = content.split('\n');
  let result: CSVRow[] = rows.map((row) => {
    return convertStringToCSVRow(row);
  });
  const firstRow = result[0];
  if (
    firstRow.firstname === header.firstname &&
    firstRow.lastname === header.lastname &&
    firstRow.title === header.title &&
    firstRow.email === header.email
  ) {
    // remove the first row if it's header
    result = result.slice(1);
  }
  return result;
}

function downloadRecords(records: CreateShadowAccountRecord[]) {
  let appliedRecords = [responseHeader];
  if (!!records && records.length > 0) {
    appliedRecords = appliedRecords.concat(records);
  }
  if (appliedRecords.length === 1) {
    return;
  }
  const csvString = appliedRecords
    .map((record) => {
      return convertRecordToCSVString(record);
    })
    .join('');
  const url = generateCVSDownloadLink(csvString);
  window.location.assign(url);
}
