import React, { useState, useCallback, useEffect } from 'react';
import {
  TableRow,
  Button,
  TableCell,
  Display,
  JustifyContent,
  Layout,
  ButtonIcon,
  SVGAsset,
  ButtonIconType,
  CoreLink,
  ButtonType,
} from 'twitch-core-ui';
import { AllowlistEntry } from '~core/clients/twitch-e2-ingest-http/twitch.fulton.example';
import { formatTimestampAsYearMonthDay } from '~core/utils/format-date';
import { ConfirmationDialog } from '~features/confirmation-dialog/confirmation-dialog';
import { useModal } from 'tachyon-modal';
import { EntityProps } from '~features/entity-list/entity-list';
import { Company, Id } from '~core/clients/rbac/code.justin.tv.devrel';
import { RBACClient } from '~core/vienna';
import { Products } from '~features/e2/product';
import { AllowlistE2Modal } from '~features/allowlist-e2-modal';

export interface PublicProps {
  onEditEntry: (entry: AllowlistEntry) => void;
  onDeleteEntry: (entry: AllowlistEntry) => void;
  withOrganization?: boolean;
}

type Props = PublicProps & EntityProps<AllowlistEntry>;

export const E2ListItem: React.FC<Props> = (props) => {
  const {
    openModal: deleteEntryOpenModal,
    id: deleteEntryId,
    active: deleteEntryActive,
    closeModal: deleteEntryCloseModal,
  } = useModal();

  const {
    openModal: editEntryOpenModal,
    id: editEntryId,
    active: editEntryActive,
    closeModal: editEntryCloseModal,
  } = useModal();

  function onDelete(confirmation: boolean) {
    if (confirmation) {
      props.onDeleteEntry(props.item);
    }
  }

  const [organization, setOrganization] = useState<Company>();
  async function getOrganization() {
    try {
      if (!!props.item.organizationId) {
        const company = await RBACClient.getCompany(new Id({ id: props.item.organizationId }));
        setOrganization(company);
      } else {
        setOrganization(undefined);
      }
    } catch {}
  }
  const getOrganizationCallback = useCallback(getOrganization, [props.item.organizationId]);

  useEffect(() => {
    getOrganizationCallback();
  }, [getOrganizationCallback, props.item.organizationId]);

  return (
    <>
      <TableRow>
        <TableCell>{props.item.clientId}</TableCell>
        <TableCell>{props.item.gameId}</TableCell>
        <TableCell>{props.item.gameName}</TableCell>
        {props.withOrganization && (
          <TableCell>
            {organization && (
              <CoreLink linkTo={`/organizations/${organization.id}`}>
                {organization.companyName}
              </CoreLink>
            )}
          </TableCell>
        )}
        <TableCell>{Products.get(props.item.product)}</TableCell>
        <TableCell>{formatTimestampAsYearMonthDay(props.item.createdAt)}</TableCell>
        <TableCell>{props.item.createdBy}</TableCell>
        <TableCell>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Around}>
            <Button onClick={editEntryOpenModal} type={ButtonType.Secondary} blurAfterClick>
              Edit
            </Button>
            <ButtonIcon
              icon={SVGAsset.Trash}
              type={ButtonIconType.Primary}
              onClick={deleteEntryOpenModal}
              blurAfterClick
            ></ButtonIcon>
          </Layout>
        </TableCell>
      </TableRow>
      <AllowlistE2Modal
        id={editEntryId}
        active={editEntryActive}
        closeModal={editEntryCloseModal}
        onSubmit={props.onEditEntry}
        entry={props.item}
        organization={organization}
      />
      <ConfirmationDialog
        id={deleteEntryId}
        active={deleteEntryActive}
        closeModal={deleteEntryCloseModal}
        text={`Are you sure you want to delete the entry for ${props.item.clientId}, ${props.item.gameId}?`}
        onClose={onDelete}
      />
    </>
  );
};
