import React from 'react';
import { ModalId, Modal, ModalSize } from 'tachyon-modal';
import {
  ModalHeader,
  ModalBody,
  FormGroup,
  Input,
  InputType,
  Layout,
  Select,
  ModalFooter,
} from 'twitch-core-ui';
import * as yup from 'yup';
import { UserInformation } from '~features/organizations/models/user-information';
import { useFormik } from 'formik';
import { ORGANIZATION_ROLES } from '~features/organizations/models/roles';
import './styles.scss';

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  onClose: (userInfo: UserInformation) => void;
  active: boolean;
  user?: UserInformation;
}

type Props = PublicProps;

const formValidation = yup.object<UserInformation>({
  firstName: yup.string().required(),
  lastName: yup.string().required(),
  title: yup.string().required(),
  role: yup.string().required(),
  twitchUserId: yup.string().required(),
});

export const ManageUserModal: React.FC<Props> = (props) => {
  const formik = useFormik<UserInformation>({
    onSubmit: onSubmit,
    initialValues: {
      firstName: props.user?.firstName || '',
      lastName: props.user?.lastName || '',
      title: props.user?.title || '',
      role: props.user?.role || '',
      twitchUserId: props.user?.twitchUserId || '',
    },
    validationSchema: formValidation,
    validateOnChange: true,
    validateOnMount: !!props.user,
  });

  function onSubmit(values: UserInformation) {
    props.onClose(values);
    formik.resetForm();
    props.closeModal();
  }

  function closeModal() {
    formik.resetForm();
    props.closeModal();
  }

  return (
    <Modal id={props.id} size={ModalSize.Large}>
      <ModalHeader
        title={props.user ? 'Edit User' : 'Add User'}
        closeButton={{ 'aria-label': 'Close', onClick: closeModal }}
      />
      <form onSubmit={formik.handleSubmit} className="manage-user-form">
        <ModalBody>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="First Name"
              error={!!formik.errors.firstName}
              errorMessage={formik.errors.firstName}
            >
              <Input
                type={InputType.Text}
                value={formik.values.firstName}
                onChange={formik.handleChange}
                name="firstName"
                onBlur={formik.handleBlur}
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Last Name"
              error={!!formik.errors.lastName}
              errorMessage={formik.errors.lastName}
            >
              <Input
                type={InputType.Text}
                value={formik.values.lastName}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                name="lastName"
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Title"
              error={!!formik.errors.title}
              errorMessage={formik.errors.title}
            >
              <Input
                type={InputType.Text}
                value={formik.values.title}
                onChange={formik.handleChange}
                name="title"
                onBlur={formik.handleBlur}
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup label="Role" error={!!formik.errors.role} errorMessage={formik.errors.role}>
              <Select
                value={formik.values.role}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                name="role"
              >
                <option value="Owner" disabled>
                  Owner
                </option>
                {ORGANIZATION_ROLES.map((role, index) => {
                  return (
                    <option value={role.name} key={index}>
                      {role.name}
                    </option>
                  );
                })}
              </Select>
            </FormGroup>
          </Layout>

          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Twitch ID"
              error={!!formik.errors.twitchUserId}
              errorMessage={formik.errors.twitchUserId}
            >
              <Input
                type={InputType.Text}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                name="twitchUserId"
                value={formik.values.twitchUserId}
              />
            </FormGroup>
          </Layout>
        </ModalBody>
        <ModalFooter
          primaryButtonProps={{
            onClick: formik.submitForm,
            children: 'Save',
            disabled: !formik.isValid,
          }}
          secondaryButtonProps={{ onClick: closeModal, children: 'Cancel' }}
        />
      </form>
    </Modal>
  );
};
