import React from 'react';
import { useFormik } from 'formik';
import { OrganizationInformation } from '~pages/organization-detail/models/organization-information';
import { CompanyType } from '~features/organizations/models/organization-type';
import {
  Layout,
  FormGroup,
  InputType,
  Input,
  Select,
  Display,
  CheckBox,
  StyledLayout,
  Background,
  JustifyContent,
  Button,
  ButtonType,
} from 'twitch-core-ui';
import * as yup from 'yup';

export interface PublicProps {
  organization: OrganizationInformation;
  onSubmit: (values: OrganizationInformation) => void;
  showContractOptions?: boolean;
}

const schema = yup.object<OrganizationInformation>({
  companyName: yup.string().required(),
  url: yup.string().required(),
  type: yup.number().required(),
  vhsContractSigned: yup.bool(),
  campaignsEnabled: yup.bool(),
});

type Props = PublicProps;

export const OrganizationForm: React.FC<Props> = (props) => {
  const formik = useFormik<OrganizationInformation>({
    initialValues: {
      companyName: props.organization.companyName || '',
      url: props.organization.url || '',
      type: props.organization.type || CompanyType.Other,
      vhsContractSigned: props.organization.vhsContractSigned || false,
      campaignsEnabled: props.organization.campaignsEnabled || false,
    },
    onSubmit: props.onSubmit,
    validationSchema: schema,
  });

  return (
    <>
      <form onSubmit={formik.handleSubmit}>
        <StyledLayout padding={2} background={Background.Base}>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Company Name"
              error={!!formik.errors.companyName}
              errorMessage={formik.errors.companyName}
            >
              <Input
                name="companyName"
                type={InputType.Text}
                value={formik.values.companyName}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Company URL"
              error={!!formik.errors.url}
              errorMessage={formik.errors.url}
            >
              <Input
                name="url"
                type={InputType.Text}
                value={formik.values.url}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Company Type"
              error={!!formik.errors.type}
              errorMessage={formik.errors.type}
            >
              <Select
                name="type"
                value={formik.values.type.toString()}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
              >
                <option value={CompanyType.Developer}>Developer</option>
                <option value={CompanyType.Publisher}>Publisher</option>
                <option value={CompanyType.Advertising}>Advertising</option>
                <option value={CompanyType.Tools}>Tools</option>
                <option value={CompanyType.Other}>Other</option>
              </Select>
            </FormGroup>
          </Layout>
          {props.showContractOptions && (
            <Layout display={Display.Flex}>
              <Layout margin={{ right: 2 }}>
                <CheckBox
                  name="vhsContractSigned"
                  label="VHS Contract Signed?"
                  checked={formik.values.vhsContractSigned}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
              </Layout>
              <Layout>
                <CheckBox
                  label="Campaigns Enabled?"
                  name="campaignsEnabled"
                  checked={formik.values.campaignsEnabled}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
              </Layout>
            </Layout>
          )}
        </StyledLayout>
        <StyledLayout
          background={Background.Alt2}
          padding={2}
          display={Display.Flex}
          justifyContent={JustifyContent.End}
        >
          <Button type={ButtonType.Primary} disabled={!formik.isValid || !formik.dirty}>
            Save Changes
          </Button>
        </StyledLayout>
      </form>
    </>
  );
};
