import React, { useState, useEffect } from 'react';
import { ModalId, Modal } from 'tachyon-modal';
import {
  ModalBody,
  ModalHeader,
  ModalFooter,
  FormGroup,
  Input,
  InputType,
  Layout,
  CoreText,
  TextType,
  SVG,
  SVGAsset,
  Display,
  JustifyContent,
  Button,
  MediaCard,
  MediaCardTitle,
  MediaCardMeta,
  MediaCardLink,
  Tag,
  InFeatureNotification,
  NotificationType,
  Select,
} from 'twitch-core-ui';
import { useFormik } from 'formik';
import * as yup from 'yup';
import { krakenClient, RBACClient } from '~core/vienna';
import { ExtensionSearchResult, KrakenExtension } from '~core/clients/kraken/models/extensions';
import {
  GetUsersByCompanyIdRequest,
  SortOrder,
  Membership,
  TransferExtensionRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  onClose: (transferExtension: TransferExtensionRequest) => void;
  companyInformation: {
    name: string;
    id: string;
  };
}

type Props = PublicProps;

interface TransferExtensionForm {
  extensionId: string;
}

const validationSchema = yup.object<TransferExtensionForm>({
  extensionId: yup.string().required(),
});

export const TransferExtensionModal: React.FC<Props> = (props) => {
  const [extension, setExtension] = useState<KrakenExtension | undefined>();
  const [errorMessage, setErrorMessage] = useState('');
  const [transferValid, setTransferValid] = useState(false);
  const [billingManagers, setBillingManagers] = useState<Membership[]>([]);
  const [selectedBillingManager, setSelectedBillingManager] = useState<string | undefined>();

  const formik = useFormik<TransferExtensionForm>({
    initialValues: {
      extensionId: '',
    },
    onSubmit: transferExtension,
    validationSchema,
  });

  useEffect(() => {
    let valid = true;

    if (!extension) {
      valid = false;
    }

    if (extension?.bits_enabled && !selectedBillingManager) {
      valid = false;
    }

    setTransferValid(valid);
  }, [extension, extension?.bits_enabled, selectedBillingManager]);

  function transferExtension(values: TransferExtensionForm) {
    searchExtension(values.extensionId);
  }

  async function handleExtensionTransfer() {
    props.onClose(
      new TransferExtensionRequest({
        companyId: props.companyInformation.id,
        extensionId: extension?.id,
        billingManagerId: selectedBillingManager,
      }),
    );
    props.closeModal();
  }

  async function searchExtension(clientId: string) {
    const result = await krakenClient.post<ExtensionSearchResult>('/v5/extensions/search', {
      limit: 1,
      offset: 0,
      searches: [
        {
          field: 'id',
          term: clientId,
        },
      ],
    });

    if (result.json.extensions?.length) {
      setExtension(result.json.extensions[0]);

      if (result.json.extensions[0].bits_enabled) {
        const billingManagerResponse = await RBACClient.getUsersByCompanyId(
          new GetUsersByCompanyIdRequest({
            id: props.companyInformation.id,
            role: 'billing_manager',
            offset: 0,
            limit: 10,
            sortBy: 'role',
            orderBy: SortOrder.ASC,
          }),
        );

        const ownerResponse = await RBACClient.getUsersByCompanyId(
          new GetUsersByCompanyIdRequest({
            id: props.companyInformation.id,
            role: 'owner',
            offset: 0,
            limit: 1,
            sortBy: 'role',
          }),
        );

        const billingManagersEligable = [
          ...billingManagerResponse.memberships,
          ...ownerResponse.memberships,
        ];

        if (!billingManagersEligable.length) {
          setErrorMessage('No Billing Managers available');
          return;
        }

        setBillingManagers(billingManagersEligable);
        setSelectedBillingManager(billingManagersEligable[0].twitchId);
      }
    } else {
      formik.setErrors({ extensionId: 'No Extension found' });
    }
  }

  return (
    <Modal id={props.id}>
      <ModalHeader
        closeButton={{ onClick: props.closeModal, 'aria-label': 'Close' }}
        title="Transfer Extension"
      />
      <ModalBody>
        {errorMessage && (
          <Layout margin={{ bottom: 2 }}>
            <InFeatureNotification
              type={NotificationType.Error}
              message={{ title: 'Error!', description: errorMessage }}
            />
          </Layout>
        )}
        <Layout margin={{ bottom: 2 }}>
          <Layout margin={{ bottom: 2 }}>
            <CoreText>
              Transfer an Extension from a User to this Organization
              <CoreText bold type={TextType.Span}>
                ({props.companyInformation.name})
              </CoreText>
            </CoreText>
          </Layout>
          <Layout margin={{ bottom: 1 }} display={Display.Flex}>
            <Layout margin={{ right: 1 }}>
              <SVG asset={SVGAsset.Communities} />
            </Layout>
            <CoreText>The User that owns the Extension must be part of this Organization</CoreText>
          </Layout>
          <Layout display={Display.Flex}>
            <Layout margin={{ right: 1 }}>
              <SVG asset={SVGAsset.BountyBoard} />
            </Layout>
            <CoreText>
              If the Extension is monetized, this Organization requires a Billing Manager
            </CoreText>
          </Layout>
        </Layout>
        <form onSubmit={formik.handleSubmit}>
          <Layout margin={{ bottom: 1 }}>
            <FormGroup
              error={!!formik.errors.extensionId}
              errorMessage={formik.errors.extensionId}
              label="Extension ID"
            >
              <Input
                type={InputType.Text}
                value={formik.values.extensionId}
                onChange={formik.handleChange}
                name="extensionId"
                onBlur={formik.handleBlur}
              />
            </FormGroup>
          </Layout>
          <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
            <Button disabled={!formik.isValid || !formik.dirty}>Search</Button>
          </Layout>
        </form>
        {extension && (
          <>
            <Layout margin={{ bottom: 2 }}>
              <MediaCard
                meta={
                  <MediaCardMeta
                    links={[<MediaCardLink key={extension.id} children={extension.id} />]}
                    title={<MediaCardTitle>{extension.name}</MediaCardTitle>}
                    actions={[
                      extension.bits_enabled ? <Tag label="Monetized" key={1} /> : undefined,
                    ]}
                  />
                }
              />
            </Layout>
            {extension.bits_enabled && (
              <FormGroup label="BIlling Manager">
                <Select
                  value={selectedBillingManager}
                  onChange={(ev) => setSelectedBillingManager(ev.currentTarget.value)}
                >
                  {billingManagers.map((bm) => (
                    <option value={bm.twitchId} key={bm.twitchId}>
                      {`${bm.firstName} ${bm.lastName} <${bm.devEmail}>`}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            )}
          </>
        )}
      </ModalBody>
      <ModalFooter
        primaryButtonProps={{
          onClick: handleExtensionTransfer,
          disabled: !transferValid,
          children: 'Transfer',
        }}
        secondaryButtonProps={{ children: 'Cancel', onClick: props.closeModal }}
      />
    </Modal>
  );
};
