import React, { useState, useEffect, useCallback } from 'react';
import extensionsListQuery from './queries/list-extensions.gql';
import getExtensionByClientIdQuery from './queries/get-extension-by-client-id.gql';
import { useLazyQuery } from '@apollo/react-hooks';
import {
  listExtensions,
  listExtensionsVariables,
  listExtensions_extensions_edges_node,
  getExtensionByClientId,
  getExtensionByClientIdVariables,
} from '~core/graphql/twitch-gql-schema';
import { EntityList } from '~features/entity-list';
import { ExtensionListItem } from '~features/extensions/components/extension-list-item';
import { DebounceTimer } from '~core/utils/debounce-timer';
import { SearchInput } from 'twitch-core-ui';

const searchDebounceTimer = new DebounceTimer(300);

enum ResultDisplayMode {
  List = 'list',
  ClientID = 'clientId',
}

export const ExtensionsListPage: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [cursors, setCursors] = useState<string[]>([]);
  const [currentCursor, setCurrentCursor] = useState<string>();
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [clientIdSearch, setClientIdSearch] = useState('');
  const [dispalyMode, setDisplayMode] = useState<ResultDisplayMode>(ResultDisplayMode.List);

  const [loadExtensions, { data }] = useLazyQuery<listExtensions, listExtensionsVariables>(
    extensionsListQuery,
    {
      onCompleted: (data) => {
        if (data && data.extensions) {
          const edgeCount = data.extensions.edges.length;
          if (edgeCount && edgeCount > 0) {
            const lastEdge = data.extensions.edges[edgeCount - 1];
            setCursors((currenctCursors) => {
              if (!currenctCursors.includes(lastEdge.cursor)) {
                currenctCursors.push(lastEdge.cursor);
              }
              return currenctCursors;
            });
            if (currentCursor !== lastEdge.cursor) {
              setCurrentCursor(lastEdge.cursor);
            }
            setTotalPages(Math.ceil(data.extensions.totalCount / 25));
          }
        }
      },
    },
  );

  const [getExtensionById, { data: clientIdExtensionData }] = useLazyQuery<
    getExtensionByClientId,
    getExtensionByClientIdVariables
  >(getExtensionByClientIdQuery);

  async function getExtensions(cursor = '') {
    loadExtensions({
      variables: {
        cursor: cursor,
      },
    });
  }
  const getExtensionsCallback = useCallback(getExtensions, [loadExtensions]);

  useEffect(() => {
    getExtensionsCallback();
  }, [getExtensionsCallback]);

  function onNextPage() {
    if (!currentCursor) {
      return;
    }
    setCurrentPage((page) => page + 1);
    getExtensionsCallback(currentCursor);
  }

  async function onPreviousPage() {
    if (!currentCursor) {
      return;
    }

    const cursorIndex = cursors.indexOf(currentCursor);

    if (cursorIndex === 0) {
      return;
    } else if (cursorIndex === 1) {
      setCurrentPage((page) => page - 1);
      await getExtensionsCallback();
      return;
    }

    setCurrentPage((page) => page - 1);
    const previousCursor = cursors[cursorIndex - 2];
    getExtensionsCallback(previousCursor);
  }

  async function searchExtension() {
    setCursors([]);
    setCurrentPage(1);
    setCurrentCursor(undefined);
    setDisplayMode(ResultDisplayMode.List);
    loadExtensions({
      variables: {
        cursor: '',
        search: searchTerm,
      },
    });
  }
  const searchExtensionsCallback = useCallback(searchExtension, [loadExtensions, searchTerm]);

  useEffect(() => {
    searchDebounceTimer.invoke(() => {
      searchExtensionsCallback();
    });
  }, [searchExtensionsCallback, searchTerm]);

  useEffect(() => {
    if (clientIdSearch.length >= 30) {
      setDisplayMode(ResultDisplayMode.ClientID);
      setCursors([]);
      setCurrentPage(1);
      setTotalPages(1);
      getExtensionById({ variables: { clientId: clientIdSearch } });
    } else {
      setDisplayMode(ResultDisplayMode.List);
    }
  }, [clientIdSearch, getExtensionById]);

  const extensions =
    dispalyMode === ResultDisplayMode.List
      ? data?.extensions?.edges.map((e) => e.node) || []
      : (clientIdExtensionData &&
          clientIdExtensionData.extension && [clientIdExtensionData.extension]) ||
        [];

  return (
    <EntityList<listExtensions_extensions_edges_node>
      items={extensions}
      columns={['Name', 'ClientID', 'Author Name', 'Insights']}
      zeroStateMessage="No Extensions found"
      rowType={ExtensionListItem}
      rowProps={{}}
      header={{
        onSearchUpdate: setSearchTerm,
        searchTerm: searchTerm,
        searchPlaceholder: 'Search Extensions',
        additionalFields: [
          <SearchInput
            placeholder="Search by Client ID"
            value={clientIdSearch}
            onChange={(ev: React.ChangeEvent<HTMLInputElement>) =>
              setClientIdSearch(ev.target.value)
            }
          />,
        ],
      }}
      cursorPagination={{
        hasNext: data?.extensions?.pageInfo.hasNextPage || false,
        hasPrevious: data?.extensions?.pageInfo.hasPreviousPage || true,
        onNextPage: onNextPage,
        onPreviousPage: onPreviousPage,
        totalPages: totalPages,
        currentPage: currentPage,
      }}
    />
  );
};
