import React, { useState, useEffect, useCallback } from 'react';
import { useParams } from 'react-router';
import { ExtensionReviewClient } from '~core/vienna';
import {
  ExtensionVersionID,
  Extension,
  SetStateReviewerRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';
import { ExtensionReviewPresentation } from '~pages/extensions-review/extension-review-presentation';
import { Layout, JustifyContent, LoadingSpinner, AlignItems } from 'twitch-core-ui';
import { PolicyAction } from '~pages/extensions-review/components/extension-review-form';

interface RouteProps {
  clientId: string;
  version: string;
}

export const ExtensionsReviewPage: React.FC = () => {
  const { clientId, version } = useParams<RouteProps>();
  const [reviewExtension, setReviewExtension] = useState<Extension>();
  const [policies, setPolicies] = useState<string[]>([]);
  const [reviewNotes, setReviewNotes] = useState<string>('');

  async function fetchExtensionReview() {
    try {
      const result = await ExtensionReviewClient.getExtension(
        new ExtensionVersionID({
          id: clientId,
          version: version,
        }),
      );
      setReviewExtension(result);
    } catch {
      toast('Sorry! Failed ot load the Extension', { type: 'error' });
    }
  }

  const fetchExtensionReviewCallback = useCallback(fetchExtensionReview, [clientId, version]);

  useEffect(() => {
    fetchExtensionReviewCallback();
  }, [fetchExtensionReviewCallback]);

  if (!reviewExtension) {
    return (
      <Layout
        fullHeight
        fullWidth
        justifyContent={JustifyContent.Center}
        alignItems={AlignItems.Center}
      >
        <LoadingSpinner />
      </Layout>
    );
  }

  function onPolicyChange(policy: string, action: PolicyAction) {
    console.log('action', action);
    console.log('policy', policy);
    if (action === PolicyAction.Added) {
      if (policies.includes(policy)) {
        return;
      }

      const newPolicies = [...policies, policy];
      setPolicies(newPolicies);
    } else {
      const policyIndex = policies.findIndex((p) => p === policy);
      if (policyIndex > -1) {
        const newPolicies = policies.filter((p) => p !== policy);
        setPolicies(newPolicies);
      }
    }
  }

  async function rejectExtension() {
    try {
      await ExtensionReviewClient.setStatePendingAction(
        new SetStateReviewerRequest({
          extensionId: clientId,
          extensionVersion: version,
          reviewReasonCodes: policies,
          reviewReason: reviewNotes,
        }),
      );
      toast('Extension has been reviewed!', { type: 'success' });
      fetchExtensionReview();
    } catch (err) {
      toast(`Failed to update extension: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function approveExtension() {
    try {
      await ExtensionReviewClient.setStateApproved(
        new SetStateReviewerRequest({
          extensionId: clientId,
          extensionVersion: version,
          reviewReasonCodes: policies,
          reviewReason: reviewNotes,
        }),
      );
      toast('Extension has been reviewed!', { type: 'success' });
      fetchExtensionReview();
    } catch (err) {
      toast(`Failed to update extension: ${(err as Error).message}`, { type: 'error' });
    }
  }

  return (
    <ExtensionReviewPresentation
      extension={reviewExtension}
      policies={policies}
      reviewNotes={reviewNotes}
      onPolicyChange={onPolicyChange}
      onReviewNotesChange={setReviewNotes}
      onApprove={approveExtension}
      onReject={rejectExtension}
    />
  );
};
