import React, { useState, useCallback, useEffect } from 'react';
import {
  GameApplication,
  ListGameApplicationsRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { usePagination } from '~features/hooks/use-pagination';
import { RBACClient } from '~core/vienna';
import { EntityList } from '~features/entity-list';
import { GameReviewListItem } from '~features/games/review/components/game-review-list-item';

const APPLICATION_LIMIT = 25;

export const GameReviewListPage: React.FC = () => {
  const [applications, setApplications] = useState<GameApplication[]>([]);
  const [totalPages, setTotalPages] = useState(0);
  const { nextPage, page, previousPage, goToPage } = usePagination();
  const [loading, setLoading] = useState(false);

  async function getGameApplications() {
    setLoading(true);
    try {
      const result = await RBACClient.listGameApplications(
        new ListGameApplicationsRequest({
          limit: APPLICATION_LIMIT,
          offset: APPLICATION_LIMIT * (page - 1),
        }),
      );
      setApplications(result.gameApplications);
      setTotalPages(Math.ceil(result.Total / APPLICATION_LIMIT));
    } catch {
    } finally {
      setLoading(false);
    }
  }

  const getGameApplicationsCallback = useCallback(getGameApplications, [page]);

  useEffect(() => {
    getGameApplicationsCallback();
  }, [getGameApplicationsCallback, page]);

  return (
    <EntityList<GameApplication>
      pagination={{
        onGoToPage: goToPage,
        onPreviousPage: previousPage,
        onNextPage: nextPage,
        currentPage: page,
        totalPages: totalPages,
      }}
      loading={loading}
      columns={[
        'Game Name',
        'Game ID',
        'Requesting Organization',
        'Owning Organization',
        'Date Created',
        'Twitch Directory Link',
        'IGDB/GB Link',
      ]}
      items={applications}
      rowType={GameReviewListItem}
      rowProps={{}}
      zeroStateMessage="No Game Application found"
      header={{}}
    />
  );
};
