import React, { useState, useCallback, useEffect } from 'react';
import { Layout, Display, JustifyContent, LoadingSpinner, AlignItems } from 'twitch-core-ui';
import { useParams, useHistory, useRouteMatch } from 'react-router';
import { OrganizationDetailPresentation } from '~pages/organization-detail/organization-detail-presentation';
import {
  Company,
  Id,
  UpdateCompanyRequest,
  AddUserToCompanyRequest,
  CreateUserRequest,
  CreateGameApplicationRequest,
  OnboardGameRequest,
  DeleteGameApplicationRequest,
  TransferExtensionRequest,
  TransferCompanyOwnershipRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { OrganizationInformation } from '~pages/organization-detail/models/organization-information';
import { toast } from 'react-toastify';
import { RBACClient, RBACAdminClient, TwitchE2IngestHTTPClient } from '~core/vienna';
import { ActionHistoryModal } from '~features/action-history-modal';
import { useModal } from 'tachyon-modal';
import { ManageUserModal } from '~features/manage-user-modal';
import { UserInformation } from '~features/organizations/models/user-information';
import { AddGameModal } from '~features/add-game-modal';
import { Game } from '~features/game-autocomplete/models/game';
import { CreateShadowAccountModal } from '~features/create-shadow-account';
import { TransferExtensionModal } from '~features/transfer-extension-modal';
import { TransferOrganizationModal } from '~features/transfer-organization-modal';
import { AllowlistE2Modal, E2ModalSchema } from '~features/allowlist-e2-modal/allowlist-e2-modal';
import { CreateAllowlistEntryRequest } from '~core/clients/twitch-e2-ingest-http/twitch.fulton.example';
import { getJwtProperty } from 'bouncer-auth-js';

export interface RouteProps {
  organizationId: string;
}

export const OrganizationDetailPage: React.FC = () => {
  const [company, setCompany] = useState<Company>();
  const params = useParams<RouteProps>();
  const history = useHistory();
  const routeMatch = useRouteMatch();

  const {
    id: activityId,
    active: activityActive,
    openModal: activityOpenModal,
    closeModal: activityCloseModal,
  } = useModal();

  const {
    id: addUserId,
    active: addUserActive,
    openModal: addUserOpenModal,
    closeModal: addUserCloseModal,
  } = useModal();

  const {
    id: addGameId,
    active: addGameActive,
    openModal: addGameOpenModal,
    closeModal: addGameCloseModal,
  } = useModal();

  const {
    id: transferExtensionId,
    openModal: transferExtensionOpenModal,
    closeModal: transferExtensionCloseModal,
  } = useModal();

  const {
    id: transferOwnershipId,
    openModal: transferOwnershipOpenModal,
    closeModal: transferOwnershipCloseModal,
    active: transferOwnershipActive,
  } = useModal();

  const {
    id: addShadowAccountId,
    openModal: addShadowAccountModal,
    closeModal: closeShadowAccountModal,
  } = useModal();

  const {
    id: allowlistE2Id,
    openModal: allowlistE2OpenModal,
    closeModal: allowlistE2CloseModal,
    active: allowlistE2Active,
  } = useModal();

  async function addUser(value: UserInformation) {
    try {
      await RBACClient.addUserToCompany(
        new AddUserToCompanyRequest({
          user: new CreateUserRequest({
            twitchId: value.twitchUserId,
            firstName: value.firstName,
            lastName: value.lastName,
            title: value.title,
          }),
          companyId: params.organizationId,
          role: value.role,
          requestingTwitchId: '',
        }),
      );
      toast('Added User!', { type: 'success' });
    } catch (e) {
      toast(`Failed to add user: ${e.message}`, { type: 'error' });
    }
  }

  async function getCompany() {
    if (params.organizationId) {
      const response = await RBACClient.getCompany(new Id({ id: params.organizationId }));
      if (response) {
        setCompany(response);
      }
    }
  }
  const getCompanyCallback = useCallback(getCompany, [params.organizationId]);

  async function updateCompany(organizationValues: OrganizationInformation) {
    if (params.organizationId) {
      const payload = new UpdateCompanyRequest({
        id: params.organizationId,
        identifier: company?.identifier,
        ...organizationValues,
      });

      try {
        const response = await RBACClient.updateCompany(payload);
        setCompany(response);
        toast('Updated Organization!', { type: 'success' });
      } catch {
        toast('Failed to update Organization', { type: 'warning' });
      }
    }
  }

  async function onGameAdd(game?: Game) {
    if (!game) {
      return;
    }

    try {
      const result = await RBACClient.createGameApplication(
        new CreateGameApplicationRequest({
          gameId: parseInt(game.id, 10),
          companyId: params.organizationId,
        }),
      );

      try {
        await RBACClient.onboardGame(
          new OnboardGameRequest({
            gameApplicationId: result.id,
          }),
        );

        toast(`Added Game to Organization!`, { type: 'success' });
        history.push(`${routeMatch.url}/games`);
      } catch (err) {
        toast(`Error Onboarding Game: ${(err as Error).message}`, { type: 'error' });

        try {
          await RBACClient.deleteGameApplication(
            new DeleteGameApplicationRequest({
              id: result.id,
              skipEmail: true,
            }),
          );
        } catch (err) {
          toast(`Error Deleting Game Application: ${(err as Error).message}`, { type: 'error' });
        }
      }
    } catch (err) {
      toast(`Can't create Game Application: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function transferExtension(extensionTransfer: TransferExtensionRequest) {
    try {
      await RBACClient.transferExtensionToCompany(extensionTransfer);
      toast('Successfully transfered the extension!', { type: 'success' });
    } catch (err) {
      toast(`Cannot transfer extension: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function transferOrganization(newRole: string, newOwnerId: string) {
    try {
      await RBACAdminClient.transferCompanyOwnership(
        new TransferCompanyOwnershipRequest({
          ownersNewRole: newRole,
          twitchIdToPromote: newOwnerId,
          companyId: params.organizationId,
        }),
      );
      toast('Successfully assigned new owner!', { type: 'success' });
    } catch (err) {
      toast(`Error transfering ownership: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function createAllowlistEntry(values: E2ModalSchema) {
    try {
      await TwitchE2IngestHTTPClient.createAllowlistEntry(
        new CreateAllowlistEntryRequest({
          ...values,
          createdBy: getJwtProperty('bouncer_auth', 'name'),
        }),
      );
      toast('Allowlisted!', { type: 'success' });
    } catch (err) {
      toast(`Failed to allowlist: ${err.message}`, { type: 'error' });
    }
  }

  useEffect(() => {
    getCompanyCallback();
  }, [getCompanyCallback]);

  if (!company) {
    return (
      <Layout
        fullHeight
        fullWidth
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        alignItems={AlignItems.Center}
      >
        <LoadingSpinner />
      </Layout>
    );
  }

  return (
    <>
      <OrganizationDetailPresentation
        company={company}
        onCompanyUpdate={updateCompany}
        onActionHistory={activityOpenModal}
        onAddUser={addUserOpenModal}
        onAddGame={addGameOpenModal}
        onTransferExtension={transferExtensionOpenModal}
        onTransferOwnership={transferOwnershipOpenModal}
        onAddShadowAccount={addShadowAccountModal}
        onAllowlistE2={allowlistE2OpenModal}
      />
      <ActionHistoryModal
        companyId={params.organizationId}
        id={activityId}
        closeModal={activityCloseModal}
        active={activityActive}
      />
      <ManageUserModal
        id={addUserId}
        closeModal={addUserCloseModal}
        active={addUserActive}
        onClose={addUser}
      />
      <CreateShadowAccountModal
        id={addShadowAccountId}
        companyId={company.id}
        closeModal={closeShadowAccountModal}
      />
      <AddGameModal
        id={addGameId}
        closeModal={addGameCloseModal}
        onClose={onGameAdd}
        active={addGameActive}
      />
      <TransferExtensionModal
        id={transferExtensionId}
        closeModal={transferExtensionCloseModal}
        onClose={transferExtension}
        companyInformation={{ id: company.id, name: company.companyName }}
      />
      <TransferOrganizationModal
        onCloseModal={transferOrganization}
        id={transferOwnershipId}
        closeModal={transferOwnershipCloseModal}
        organizationId={params.organizationId}
        active={transferOwnershipActive}
      />
      <AllowlistE2Modal
        id={allowlistE2Id}
        closeModal={allowlistE2CloseModal}
        active={allowlistE2Active}
        onSubmit={createAllowlistEntry}
        organization={company}
      />
    </>
  );
};
